/**
 *  Loopuino RF code for Arduino
 *
 *  The RF code is adapted from an example by Maurice Ribble from
 *  http://www.glacialwanderer.com/hobbyrobotics
 *
 *  It uses SoftwareSerial now for sending/recieving via serial port.
 *  It uses a checksum and a network signature to prevent it from receiving garbage.
 *
 *  Each packet has the following format:
 *  BYTE#1  Sync-Byte 0xF0
 *  BYTE#2  Network-Signature (to let multiple Loopuinos work on the same freq. change this)
 *  BYTE#3  8 higher bits of the value
 *  BYTE#4  8 lower bits of the value
 *  BYTE#5  checksum for the value
 *  
 *  @author Markus Konrad <mako@mako-home.de>, Tobias Schultze
 */

#define NETWORK_SIG_SIZE 1
#define VAL_SIZE         2
#define CHECKSUM_SIZE    1
#define PACKET_SIZE      (NETWORK_SIG_SIZE + VAL_SIZE + CHECKSUM_SIZE)


const byte network_sig = 0x8F;  // Network signature - to let multiple Loopuinos work on the same freq. change this

/**
 *  Sends an unsigned int over the RF network
 *  @param value to send
 */
void sendUInt(unsigned int val)
{
  byte checksum = calcChecksum(val);
  
  //Send the whole packet:
  mySerial.print(0xF0, BYTE);  // This gets reciever in sync with transmitter
  mySerial.print(network_sig, BYTE);
  mySerial.print((byte)(val >> 8), BYTE);  //higher bits first
  mySerial.print((byte)val, BYTE);
  mySerial.print(checksum, BYTE); //CHECKSUM_SIZE
}

/**
 *  Receives an int over the RF network
 *  @param pointer to a boolean that indicates whether a valid packet has been fetched
 *  @return integer value that has been fetched
 */
unsigned int rcvInt(boolean* gotPacket)
{
  ///boolean gotPacket = false;
  unsigned int val;     // Value of the unsigned int
  byte c = 0;           // Current byte
  
  //while(!gotPacket)
  //{
    byte checksum;
    
    c = mySerial.read();
    
    if (c == network_sig)
    {
        val      = ((unsigned int)mySerial.read())*256;
        val      += mySerial.read();
        checksum =  mySerial.read();
        
        if (checksum != calcChecksum(val))
        {
          *gotPacket = false;
        }
        else
        {
          *gotPacket = true;
        }
    }
  //}
  return val;
}

/**
 *  Calculate the checksum of an unsigned integer
 *  @param value
 *  @return checksum
 */
byte calcChecksum(unsigned int val)
{
  return ((val/256) ^ (val&0xFF));
}
