/**
 *  Loopuino main code for Arduino
 *
 *  This file includes the code for the receiver and the transmitter. Compile the code
 *  for the right target by defining the right macro in lines 15/16
 *  
 *  The RF code is adapted from an example by Maurice Ribble from
 *  http://www.glacialwanderer.com/hobbyrobotics
 *  
 *  @author Markus Konrad <mako@mako-home.de>, Tobias Schultze
 */
 
/**
 *  The Loopuino receiver device sends commands (e.g. "button pressed" or "potentiometer
 *  value changed to x") to the host application (SuperCollider) via USB serial port and
 *  listens for input commands. Each command ends with a line-feed (0x0A).
 *  The following commands are implemented:
 *
 *  OUTPUT (to SuperCollider):
 *  b <0/1>     Button down/up
 *  s <0-1023>  Potentiometer value changed to x
 *
 *  INPUT (from SuperCollider):
 *  l<0-7>      Turn on LED x (NO SPACE BETWEEN "l" AND THE VALUE)
 */


//For which target do we compile the code? Uncomment the right one here.
//Only *one* should be defined of the following:
//#define TRANSMITTER
#define RECEIVER

//Send Debug-messages to the serial console
#define DEBUG

//We need SoftwareSerial for RF communication and hw-serial for USB
#include "SoftwareSerial.h"

//Set the communication rates
#define SERIAL_RATE 115200
#define SOFTSERIAL_RATE 1200

//Define the Pins for the RF communication
#define RX_PIN 2
#define TX_PIN 3

//Loop-Record button:
#define BTN_PIN  4

//LED row:
#define LED_PIN_MIN 5
#define LED_PIN_MAX 12
#define LED_DEBUG_PIN  13
#define LED_PIN_SIZE (LED_PIN_MAX - LED_PIN_MIN + 1)
#define LED_LIGHT_MS 100

//Arduino analog Pin for the potentiometer
#define POT_PIN  2

//Create SoftwareSerial object
SoftwareSerial mySerial = SoftwareSerial(RX_PIN, TX_PIN);

//Initialize variables
#ifdef TRANSMITTER
  unsigned int potVal = 0;
  unsigned int potValOld = 0;
#endif

#ifdef RECEIVER
  unsigned int btnVal = 0;
  unsigned int btnValOld = 0;
  
  //debounce-timer:
  unsigned long btnDebounceMS = 0;          //VERY important: Use UNSIGNED LONG for milliseconds!
  
  //This is an array that stores for each LED when it has been activated
  unsigned long ledLightMS[LED_PIN_SIZE];   //VERY important: Use UNSIGNED LONG for milliseconds!
#endif

/**
 *  This function is called every time Arduino boots up.
 */
void setup()
{
  #ifdef RECEIVER
    //Define the Pin-modes
    pinMode(RX_PIN, INPUT);
    pinMode(BTN_PIN, INPUT);    
    
    #ifdef DEBUG
      pinMode(LED_DEBUG_PIN, OUTPUT);
      digitalWrite(LED_DEBUG_PIN, LOW);
    #endif
    
    //This is the whole LED row:
    for (int i = 0; i < LED_PIN_SIZE; i++)
    {
      pinMode(LED_PIN_MIN + i, OUTPUT);
      ledLightMS[i] = 0;
    }
    
    //Intialize serial interaces:
    Serial.begin(SERIAL_RATE);          //USB  for sending messages to SuperCollider
    mySerial.begin(SOFTSERIAL_RATE);    //RF-Receiver
  #endif
  
  #ifdef TRANSMITTER
    //Define the Pin-modes
    pinMode(TX_PIN, OUTPUT);
    pinMode(POT_PIN, INPUT);
    
    //Intialize serial interaces:
    mySerial.begin(SOFTSERIAL_RATE);    //RF-Sender
      
    #ifdef DEBUG
      Serial.begin(SERIAL_RATE);    //USB for debugging only
    #endif
  #endif
}

#ifdef TRANSMITTER
/**
 *  Infinite loop that is executed all the time when the TRANSMITTER Arduino is on
 *  Very simple: Just read a value from the potentiometer and send it via the RF-transmitter
 */
void loop()
{
  potVal = analogRead(POT_PIN);    // read the value from the sensor
  
  if (abs(potVal - potValOld) > 3)  //the value has changed
  {
    #ifdef DEBUG
      Serial.println(potVal);
    #endif
    
    sendUInt(potVal);  //send it!
    
    potValOld = potVal;
  }
}
#endif //TRANSMITTER

#ifdef RECEIVER
/**
 *  Infinite loop that is executed all the time when the RECEIVER Arduino is on
 *  Receives sensor values via RF, handles the button state and controls the LED row
 */
void loop()
{
  /*
  //Knight rider:
  for (int i = LED_PIN_MIN; i <= LED_PIN_MAX; i++)
  {
    digitalWrite(i, HIGH);
    delay(50);
    digitalWrite(i, LOW);
    delay(50);
  }
  */
  
  handleRF();
  handleBtn();
  handleLEDs();
  controlLEDs();
}

/**
 *  Receive values via RF and print a message to the serial port so that SuperCollider
 *  can read it.
 */
void handleRF()
{
  boolean gotPacket = false;
  unsigned int val = rcvInt(&gotPacket);
  
  if (gotPacket)
  {
    Serial.print("s ");
    Serial.println(val);
  }
}

/**
 *  Read the button state in the digital pin and send a message via serial port
 *  to SuperCollider.
 */
void handleBtn()
{
  btnVal = digitalRead(BTN_PIN);
  
  //The button state is reported when the debounce-timer is through and the
  //button state has been changed
  if (millis() - btnDebounceMS > 20 && btnVal != btnValOld)
  {
    btnDebounceMS = millis();  //set the debounce-timer
    
    if (btnVal == HIGH)
    {
      #ifdef DEBUG
        digitalWrite(LED_DEBUG_PIN, HIGH);
      #endif
      Serial.println("b 1");
    } 
    else
    {
      #ifdef DEBUG
        digitalWrite(LED_DEBUG_PIN, LOW);
      #endif
      Serial.println("b 0"); 
    }
     
    btnValOld = btnVal;
  } 
}

/**
 *  Receives a command from Serial to turn on a LED
 *  Format: l<Value>\n
 *          <Value> is the LED-Number and can be between 1 and 99
 */
void handleLEDs()
{
  const int READ_BUF_SIZE = 3;
  
  char readBuf[READ_BUF_SIZE];
  char curChar = 0;
  int readIdx = 0;
  int ledNum = -1;
  
  if (Serial.available() > 0)
  {
    //Read the bytes into the buffer until it's full or stop when we receive -1 or 10 (newline)
    while (readIdx < READ_BUF_SIZE && curChar != 0xFFFFFFFF && curChar != 10)
    { 
      curChar = (char) Serial.read();
      readBuf[readIdx] = curChar;
      
      readIdx++;
    }
    
    if (readIdx >= READ_BUF_SIZE && readBuf[0] == 'l')
    {
      ledNum = atoi(&readBuf[1]);
      if (ledNum > 0)
      {
        ledNum--;  //The values start with "1" but the Pins with "0"
        
        #ifdef DEBUG
          Serial.print("Turned on LED ");
          Serial.println(ledNum);
        #endif
        
        lightUpLED(ledNum);
      }
    }
    
    Serial.flush();
  }
}

/**
 *  Turn on the LED with number ledNum
 *  @param LED number (0 to LED_PIN_SIZE)
 */
void lightUpLED(int ledNum)
{
  if (ledNum >= LED_PIN_SIZE)
    return;
  
  digitalWrite(LED_PIN_MIN + ledNum, HIGH);  //turn on the LED
  ledLightMS[ledNum] = millis();             //set the timer
  
//  Serial.print("ledNum: ");
//  Serial.println(ledNum);
//  Serial.print("ledLightMS[ledNum]: ");
//  Serial.println(ledLightMS[ledNum]);
}

/**
 *  Turn off LEDs after some time
 */
void controlLEDs()
{
  //walk through the array of LED timers
  for (int i = 0; i < LED_PIN_SIZE; i++)
  {
    //turn of a LED when it is activated and the time is up
    if (ledLightMS[i] != 0 && millis() - ledLightMS[i] > LED_LIGHT_MS)
    {
      digitalWrite(LED_PIN_MIN + i, LOW);
      ledLightMS[i] = 0;
      
      #ifdef DEBUG
        Serial.print("Turned off LED ");
        Serial.println(i);
      #endif
    }
  }
}
#endif //RECEIVER


