/**
 *  LoopStation class
 *
 *  Loopuino class that handles all the audio and buffer stuff
 *  
 *  @author Markus Konrad <mako@mako-home.de>
 */
LoopStation {
  var s;            //Server
  var grp;          //Group where the synths reside on the server
  
  var bufLength;    //max. buffer length
  var < bufs;       //List of buffers (each track has one buffer)
  var < curBuf;     //current buffer index
  var masterLoop;   //boolean: has the master loop (first loop) been recorded?
  
  var < playInput;  //playInput synth
  var < loopPlayer; //List of loopPlayer synths (one for each buffer)
  var < loopRec;    //loopRec synth
  var < loopPhasor; //loopPhasor synth
  
  var fxDelay;      //fxDelay synth
  
  var < loopTime;   //how long is the master loop?
  var loopStart;    //when did we start to record the loop?
  var stopDelay;    //a small delay for the button
  
  var oscrLEDsync;  //OSCrepsonderNode for LED synchronizing
  var leds;         //LoopuinoLED-object
  var ledsPaused;   //boolean: Did we pause the LEDs for synchronizing?
  
  var bars;         //Bars in a barRate
  var barRate;
  var speed;        //Speed of the loop
  
  var < phasorBus;  //audio-rate bus for the phasor

  /**
   *  Create a new LoopStation
   *  @param server
   *  @param buffer length
   *  @param LoopuinoLED object
   *  @param group
   */
  *new { arg server, length, loopuinoLED, group;
    ^super.new.init(server, length, loopuinoLED, group);
  }
  
  /**
   *  Initialize a new LoopStation
   *  @param server
   *  @param buffer length
   *  @param LoopuinoLED object
   *  @param group 
   */
  init { arg server, length, loopuinoLED, group;
    s = server;
    bufLength = length;
    leds = loopuinoLED;
    grp = group;
    bufs = List[];
    loopPlayer = List[];
    
    this.addTrack;    //add the buffer for the master track
    
    phasorBus = Bus.audio(s, 1);  //create audio-rate bus for the phasor
    
    //create OSCresponderNode that receives a trigger from the loopPhasor-synth
    //each time the loop is at the beginning. This is for unpausing the LEDs when
    //they have been stopped after changing the timing or speed
    oscrLEDsync = OSCresponder(s.addr,'/tr',{ arg time,responder,msg;
      if (ledsPaused == 1) {
        if (speed >= 0) {
          leds.curLED = 0;  //start at the beginning when we have a positive speed
        } {
          leds.curLED = bars - 1; //start at the end when we have a negative speed
        };
        
        leds.task.resume;
        ledsPaused = 0;
        
        ("Unpaused LEDs").postln;
      };
    }).add;
    
    //create the basic synths
    playInput = Synth.head(grp, \playInput, [\in, 0, \out, 0]); //for playing back the input
    fxDelay = Synth.tail(grp, \fxDelay, [\delaytime, 0]);   //and the delay effect
    
    //init some values
    loopPhasor = nil;
    loopRec = nil;
    loopTime = 0;
    stopDelay = 0.1; //a small delay for recording might be caused by button handling
    bars = 4;
    barRate = 4;
    speed = 1;
    
    this.clear;
  }

  /**
   *  Create a new track by adding a new buffer to the list
   */
  addTrack {
    bufs.add(Buffer.alloc(s, 44100 * bufLength, 1));
    curBuf = bufs.size - 1;
  }
  
  /**
   *  Remove the last track and delete the buffer
   */
  undo {
    var buf, l;
    
    if (bufs.size <= 1) {
      this.clear; //this was the last track -> clear everything
    } {
      fork {
        l = loopPlayer.pop;
        l.free;
      
        buf = bufs.pop;
        buf.free;
      };
    };
  }
  
  /**
   *  Clear all the buffers and synths and create a new empty track
   */
  clear {
    if (leds != nil) {
      leds.stop;
    };
    
    masterLoop = 1;
    ledsPaused = 0;
    
    bufs.do { arg buf; buf.free; };
    bufs.free;
    
    loopPlayer.do { arg l; l.free; };
    loopPlayer.free;
    
    loopPhasor.free;
    loopPhasor = nil;
    
    bufs = List[];
    this.addTrack;
  }
  
  /**
   *  Change the speed
   *  @param new speed
   */
  setSpeed { arg spd;
    speed = spd;
    
    if (loopPhasor != nil) {
      loopPhasor.set(\speed, speed);  //set the speed to the phasor synth
    };
    
    if (leds != nil) {
      if (speed != 0) {
        leds.dur_(loopTime * (bars / barRate) * speed.reciprocal);  //Set new LED duration
      } {
        leds.dur_(0);
      };
      
      this.resyncLEDs;  //we changed something in the timing -> resync the LEDs!
    };
  }
  
  /**
   *  Change the bars which means change the loop-time
   *  @param bars
   *  @param barRate
   */
  setBars { arg b, bRate;
    bars = b;
    barRate = bRate;
    
    if (loopPhasor != nil) {
      loopPhasor.set(\looptime, loopTime * bars / barRate);  //set the looptime of the phasor synth
    };
    
    if (leds != nil) {
      leds.dur_(loopTime * (bars / barRate) * speed.reciprocal); //Set new LED duration
      leds.bars_(bars);
      
      this.resyncLEDs;  //we changed something in the timing -> resync the LEDs!
    };
  }
  
  /**
   *  Set the initial loop time of the master loop
   *  @param new loop time
   */
  setLoopTime { arg time;
    loopTime = time;
    
    if (loopPhasor != nil) {
      loopPhasor.set(\looptime, loopTime);  //set the looptime of the phasor synth
    };
    
    if (leds != nil) {
      leds.dur_(loopTime);  //Set the LED duration
    };
  }
  
  /**
   *  Set the delay time of the delay effect
   *  @param new loop time
   */
  setDelay { arg delay;
    fxDelay.set(\delaytime, delay);
  }
  
  /**
   *  Resynchronize LEDs.
   *  This function is called every time we change the timing of the loop.
   *  It just pauses the task that plays the LEDs so that it can be resumed
   *  when we receive a trigger from the loopPhasor synth at the beginning of a
   *  new loop.
   */
  resyncLEDs {
    leds.task.pause;
    ledsPaused = 1;
    
    ("Paused LEDs").postln;
  }
  
  /**
   * Start recording
   */
  startRec {
    fork {
      if (masterLoop != 1) {
        //it's not the first loop so we need to add a track
        this.addTrack;
      } {
        //it's the first loop so we need to create the phasor first
        loopPhasor = Synth.after(playInput, \loopPhasor, [\phasorBus, phasorBus.index, \bufnum, bufs[0].bufnum, \looptime, bufLength]);
      };
      
      //start the recorder and remember the time when we did so
      loopRec = Synth.after(loopPhasor, \loopRec, [\bufnum, bufs[curBuf].bufnum, \phasorBus, phasorBus]);
      loopStart = Main.elapsedTime;
    }
  }
  
  /**
   *  Stop recording
   *  @return Master loop time
   */
  stopRec {
    loopRec.free; //stop the recorder

    //add a new loopPlayer-synth that plays the new track
    loopPlayer.add(Synth.after(loopPhasor, \loopPlayer, [\out, 0, \phasorBus, phasorBus.index, \bufnum, bufs[curBuf].bufnum, \speed, speed]);
    
    if (masterLoop == 1) {  //it's a master loop so set the master loop time
      this.setLoopTime(Main.elapsedTime - loopStart - stopDelay);
      
      if (leds != nil) {
        leds.start(bars, loopTime); //Start the LEDs
      };
      
      ("Set looptime for the first time").postln;
      
      masterLoop = 0;
    });
    
    ^loopTime;
  }
  
  /**
   *  Self-destruction
   */
  quit {
    bufs.do { arg buf; buf.free; };
    bufs.free;
    
    loopPlayer.do { arg l; l.free; };
    loopPlayer.free;
  
    playInput.free;
    loopPhasor.free;
    loopRec.free;
    fxDelay.free;
    
    oscrLEDsync.free;
    
    if (leds != nil) {
      leds.quit;
    };
  }
}
