/**
 *  Loopuino main class
 *
 *  Handles the GUI stuff and interactions
 *
 *  @author Markus Konrad <mako@mako-home.de>
 */
Loopuino
{
  var s;      //SC-Server-Object

  var ard;          //use Arduino?
  var comm;         //LoopuinoComm-Object
  var leds;         //LoopuinoLED-Object
  var loopStation;  //LoopStation-Object
  
  var oldVal;       //old sensor value
  
  var loopTime;     //how long this loop is
  
  var grp;          //group for all synths
  
  var wnd;          //Window-Object
  var btnCtrl;      //Array of "Control this!"-Buttons
  var curControl;   //Which value is currently controlled by Arduino?
  var tempoSpec;    //for mapping of the tempo-values
  var tempoSlider;  //tempo slider GUI object
  var delaySpec;    //for mapping of the delay-values
  var delaySlider;  //delay slider GUI object
  var barSpec;      //for mapping of the bar-values
  var barSlider;    //bar slider GUI object
  var barTextBox;   //bar textbox GUI object
  var btnUndo;      //"Undo"-button
  var btnRecAction; //Function to be evaluated when pressing the "Record"-button
  var btnRec;       //"Record!"-button
  var btnPressed;   //Saves the time when the button was pressed

  /**
   *  Create new Loopuino object
   *  @param buffer length in seconds
   *  @param serial port (leave empty when not using Arduino)
   */
  *new { arg bufLength = 30, ardUSB = "";
    ^super.new.init(bufLength, ardUSB);
  }
  
  /**
   *  Intialize new Loopuino object
   *  @param buffer length in seconds
   *  @param serial port (leave empty when not using Arduino)
   */      
  init { arg bufLength, ardUSB;
    s = Server.default;
    
    grp = Group.basicNew(s, 1); //take this group for all the synths
  
    if (ardUSB != "") {
      ard = 1;
      comm = LoopuinoComm(ardUSB, 115200);  //take the highest baud-rate supported by USB
      leds = LoopuinoLED(comm);
      oldVal = -1;
    } {
      ard = 0;  //don't use Arduino
    };
    
		{
			this.sendSynthDefs; //create the synths on the Server
			s.sync;
			loopStation = LoopStation(s, bufLength, leds, grp);
		}.fork;

    this.createGUI;
    
    if (ard != 0) {
      btnPressed = 0;
      this.setArdActions; //set Arduino actions when receiving messages from Arduino
    };
    
    wnd.front;  //show the GUI
  }
  
  /**
   *  Create synths on the server
   */
  sendSynthDefs {
    //Synth just for playing the input
    SynthDef(\playInput, {
      arg in, out, gate=1, amp=1;

      Out.ar(out,
        Pan2.ar(
          SoundIn.ar(in) * amp,
          0.0,
          EnvGen.kr(Env.cutoff(0.1, 1), gate, doneAction: 2);
        )
      );
    }).send(s);
    
    //Synth for delay effect (must be after the loop players in the execution order!)
    SynthDef(\fxDelay, {arg out = 0, delaytime = 0.0;
      var input, effect; 

      input = In.ar(0,1);
      effect = DelayC.ar(input, 1, delaytime); //max delay of one second
      Out.ar(out, Pan2.ar(effect, 0.0));
    }).send(s);
    
    //Synth that creates a phasor which loops through the buffer
    //Writes the phasor-output to a audio-rate bus "phasorBus"
    //Also sends a trigger-message via OSC each time the phasor has hit the beginning of the loop
    SynthDef(\loopPhasor, {
      arg phasorBus, bufnum, speed=1, looptime=1;
      
      var playDir = (((-1 * speed) / speed.abs) + 2).mod(3); //change the direction on neg. speeds
      var playFreq = (looptime  / speed.abs).reciprocal;
      var playDur = BufRateScale.ir(bufnum) * 44100 * looptime;
      var phasor = VarSaw.ar(playFreq, 0, playDir).range(0, playDur);
      
      var trig = HPZ1.ar(phasor); 
      SendTrig.ar(trig, 0, 1);
      
//      Out.kr(trigsBus, trigs);
//      Out.kr(trigBus, trig);
      Out.ar(phasorBus, phasor);
    }).send(s);

    //Synth that plays a buffer
    //Takes the phasor-output from loopPhasor via the phasorBus
    SynthDef(\loopPlayer, {
      arg out, phasorBus, bufnum=0, gate=1, amp=1;

      var sig = BufRd.ar(1, bufnum, In.ar(phasorBus, 1)) * amp;

      Out.ar(out,
        Pan2.ar(
          sig,
          0.0,
          EnvGen.kr(Env.cutoff(0.1, 1), gate, doneAction: 2);
        )
      );
    }).send(s);
    
    //Synth that records audio into a buffer
    //Takes the phasor-output from loopPhasor via the phasorBus to get the current position in the buffer
    SynthDef(\loopRec, {
      arg in, bufnum, phasorBus;
      var phase = In.ar(phasorBus, 1);

      BufWr.ar(SoundIn.ar(in), bufnum, phase, 0, BufFrames.kr(0));
    }).send(s);
  }
  
  /**
   *  Create GUI elements
   */
  createGUI {
    //{
      //create window
      wnd = JSCWindow.new("Loopuino GUI", Rect(128, 64, 500, 300));
      
      //create "Control this!"-Buttons
      btnCtrl = Array.new;
      curControl = 0;
      3.do { arg i;
        var btn = JSCButton.new(wnd, Rect(350, 10 + (i * 30), 90, 25));
        btn.states = [
          [ "Control this!" ]
        ];
        btn.action = {
          curControl = i;
        };
        
        btnCtrl = btnCtrl.add(btn);
      };
      
      //create tempo-slider
      tempoSpec = ControlSpec(-4, 4, \lin, 0.1, 1.0);
      tempoSlider = EZSlider(wnd, Rect(10,10,300,20), "Tempo", tempoSpec);
      tempoSlider.action_({ arg sld;
        var val = sld.value;
         loopStation.setSpeed(val);
         ("Set to speed: " ++ val).postln;
      });
      
      //create bar-control
      barSpec = ControlSpec(0, 10, \lin, 1, 6);
      barSlider = EZSlider(wnd, Rect(10,40,254,20), "Bars", barSpec, numberWidth: 0);
      barTextBox = JSCTextField(wnd, Rect(262,40,45,20));
      barTextBox.string_("4/4");
      barSlider.action_({ arg sld;
         var barRate = 4, bars = sld.value.asInteger - 2;

         if (bars < 1, {  //if less then 1/4 bars, play 1/2^(bar-1)
           barRate = barRate * (2 ** (bars - 1).abs);
           bars = 1;
         });

         barTextBox.string_(bars.asString ++ "/" ++ barRate.asString);

         loopStation.setBars(bars, barRate);
         ("Set to bars: " ++ bars ++ "/" ++ barRate).postln;
      });
      
      //create delay-fx-slider
      delaySpec = ControlSpec(0, 1, \lin, default: 0.0);
      delaySlider = EZSlider(wnd, Rect(10,70,300,20), "Delay", delaySpec);
      delaySlider.action_({ arg sld;
        var val = sld.value;
         loopStation.setDelay(val);
         ("Set to delay: " ++ val).postln;
      });
      
      //create undo-button
      btnUndo = JSCButton.new(wnd, Rect(150, 100, 120, 30));
      btnUndo.states = [
        [ "Undo last loop" ]
      ];
      
      btnUndo.action = {
        loopStation.undo;
        ("Cleared last loop").postln;
      };
      
      //create record-button
      btnRecAction = { arg state;
        if (state == 1, {
          loopStation.startRec;
          "now recording...".postln;
         },{
           loopTime = loopStation.stopRec;
          ("stopped recording after " + loopTime + " seconds.").postln;
         });
      
        state;
      };
      
      btnRec = JSCButton.new(wnd, Rect(10, 100, 120, 30));
      btnRec.states = [
        [ "Record!" ],
        [ "Stop!" ]
      ];
      btnRec.action = { arg button;
         var state = button.value;
         
         btnRecAction.value(state);
      };
      
      //set close-action
      wnd.onClose = {
        this.quit;
      };
    //}
  }
  
  /**
   *  Set actions, that shall be executed when we receive message from Arduino
   */
  setArdActions {
    comm.action = { arg action, val;
      
      switch (action,
        "b", {  //button action
              comm.send("l 0");
              
              if (val == 1) {
                ("Button down!").postln;
                btnPressed = Main.elapsedTime;
                
                btnRecAction.value(1);
                btnRec.value_(1);
              } {
                ("Button up after " + (Main.elapsedTime - btnPressed)).postln; 
                
                //Stop recording:
                btnRecAction.value(0);
                btnRec.value_(0);
                
                //Clear the loop, if we switched very quickly:
                if (Main.elapsedTime - btnPressed < 0.25) {
                  btnUndo.action.value; //first time: the recorded "schnipsel" of less then 0.25 sec.
                  btnUndo.action.value; //the real loop we want to clear
                }
              }
        },
        "s", {  //sensor change
          var mappedVal;
          //("Sensor reports " + val).postln; 
          
          //we get values between 0 and 1023
          val = val / 1023;
          
          //if ((val > 0) && (val < 1) &&
          //  ((oldVal == -1) || ((oldVal - val).abs <= 0.05)))
          if ((val > 0) && (val < 1))
          {
            if (curControl == 0) {  //control the tempo
              mappedVal = tempoSpec.map(val);
              tempoSlider.valueAction_(mappedVal);
            } {
              if (curControl == 1) {  //control the bars
                mappedVal = barSpec.map(val);
                barSlider.valueAction_(mappedVal);
              } {
                mappedVal = delaySpec.map(val); //control the delay
                delaySlider.valueAction_(mappedVal);
              };
            };
            
            oldVal = val;
          }
        }
     );
    };
  }
  
  /**
   *  Self-destruction
   */
  quit {
    loopStation.quit;
    if (ard != 0) {
      comm.close;
    };
  }
}
