/**
 *  Loopuino communication classes
 *
 *  Handles communication with Arduino via the SerialPort class.
 *  It's based on the Arduino-Quark.
 *
 *  @author Markus Konrad <mako@mako-home.de>
 */
 
/**
 *  LoopuinoComm main class
 *  Just defines a method for sending messages to Arduino
 */
LoopuinoComm : Arduino
{
  /**
   *  Method that returns the parser we use when we receive messages on the
   *  serial port
   */
	*parserClass {
		^LoopuinoCommParser
	}
	
	/**
	 *  A method for sending messages to arduino
	 *  The messages have the simple format: message-string<line-feed>
	 */
	send { arg str;
	  var ascii = str.ascii;
	  fork {
	    ascii.do({ arg c;
    	  port.put(c);
    	});
	    port.put(10); //put a line-feed "\n" at the end
	  }
	}
}

/**
 *  Class that parses incoming messages.
 *  This is basically the same as the ArduinoParserSMS class but it fixes an error
 *  in the parseByte method (expects a line-feed and NOT a carriage return).
 */
LoopuinoCommParser : ArduinoParser
{
	var msg, msgArgStream, state;

  /**
   *  Reads a message from the SerialPort into a buffer
   */
	parse {
		msg = Array[];
		msgArgStream = CollStream();
		state = nil;
		loop { this.parseByte(port.read) };
	}

  /**
   *  Called when we received a message
   */
	finishArg {
		var msgArg = msgArgStream.contents; msgArgStream.reset;
		if (msgArg.notEmpty) {
			if (msgArg.first.isDecDigit) {
				msgArg = msgArg.asInteger;
			};
			msg = msg.add(msgArg);
		}
	}
	
	/**
	 *  Parses a byte. Recognizes a line-feed as the end of a message
	 */
	parseByte { | byte |
			if (byte === 10) {
					// CR/LF encountered
					// dispatch message
					this.finishArg;
					if (msg.notEmpty) {
						this.dispatch(msg);
						msg = Array[];
					};
					state = nil;
			} {
				if (byte === 32) {
					// eat them spaces
					state = 32;
				} {
					// anything else
					if (state == 32) {
						// finish last arg
						this.finishArg;
						state = nil;
					};
					// add to current arg
					msgArgStream << byte.asAscii;
				}
			}
	}
}

// EOF
