/**
 *  LoopuinoLED classes.
 *
 *  Sends messages to Arduino to let the LEDs blink in rhythm.
 *
 *  @author Markus Konrad <mako@mako-home.de>
 */
LoopuinoLED {
  var comm;       //LoopuinoComm object
  var clk;        //Clock
  var < task;     //the Task-object
  var <> bars;    //number of bars
  var < dur;      //overall duration
  var <> curLED;  //LED# between 0 and 7
  var direction;  //In which direction are the LEDs turned on? 1 is up, -1 is down, 0 means it will stay in place
  
  /**
   *  Create a new LoopuinoLED object
   *  @param LoopuinoComm object
   */
  *new { arg serialPort;
    ^super.new.init(serialPort);
  }
  
  /**
   *  Initialize a new LoopuinoLED object
   *  @param LoopuinoComm object
   */
  init { arg serialPort;
    var b;
    comm = serialPort;
    
    clk = TempoClock.new;
    
    this.reset;
    
    //This creates a new task that will execute its inner function again and
    //again. It waits between the execution of the function for 1 bar
    task = Task.new({
		  inf.do({  //infinity party!
		    b = bars;
		    b.do( //For each bar make the following:
		      comm.send("l" ++ (curLED + 1)); //Send a message "l<LED-Number>" to Arduino
		      curLED = (curLED + (direction * 1)).mod(b); //increment the LED number
		      
		      (1 / b).wait; //wait for 1 bar
		    );
		  });
    });
  }
  
  /**
   *  Reset to default values
   */
  reset {
    bars = 4;
    clk.tempo_(1.0);
    curLED = 0;
    direction = 1;
  }
  
  /**
   *  Set new duration
   *  @param new duration
   */
  dur_ { arg d;
    if (d < 0) {
      direction = -1;
    } {
      if (d == 0) {
        direction = 0;
        d = 1;
      } {
        direction = 1;
      };
    };
    
    dur = d.abs;
    ("dur: " ++ dur).postln;
    clk.tempo_(dur.reciprocal); // 1/dur is the new tempo for the task
  }
  
  /**
   *  Start blinking!
   */
  start { arg b, d;
    this.reset;
    this.dur_(d);
    bars = b;
    task.play(clk);
  }
  
  /**
   *  Stop blinking
   */
  stop {
    task.stop;
  }
  
  /**
   *  Self-destruction
   */
  quit {
    this.stop;
    task.free;
    clk.free;
  }
}

