//
//  AudioController.m
//  iPhoneAudio
//
//  Created by Markus Konrad on 11.11.10.
//  Copyright 2010 __MyCompanyName__. All rights reserved.
//

#import "AudioController.h"

// partly taken from libpd
static OSStatus renderCallback(void *inRefCon, AudioUnitRenderActionFlags *ioActionFlags, const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber, UInt32 inNumberFrames, AudioBufferList *ioData)
{
    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
    
	// Get a reference to the object that was passed with the callback
	// In this case, the AudioController passed itself so
	// that you can access its data.
	AudioController *audioCtrl = (AudioController*)inRefCon;
    
    // Get the remote io audio unit to render its input into the buffers
    // 1 == inBusNumber for mic input
    if (kNumInputChannels > 0) {
        AudioUnitRender(audioCtrl->audioUnit, ioActionFlags, inTimeStamp, 1, inNumberFrames, ioData);
    }
    
    short *shortBuffer = (short *) ioData->mBuffers[0].mData;
    

    // tick will generate new values
    audioCtrl->sine->tick(*audioCtrl->frames, 0);
    audioCtrl->noise->tick(*audioCtrl->frames, 1);
    
    // calculate buffer length
    int bufLength = inNumberFrames * kNumOuputChannels;
    
	// Loop through the callback buffer, copy the output from the STK to the buffer
	for (UInt32 i = 0; i < bufLength; i++) {
        shortBuffer[i] = (short)((*audioCtrl->frames)[i] * audioCtrl->channelVol[i%2] * audioCtrl->masterVolume * 32767.0f);
    }

    [pool drain];

	return noErr;
}

// the interrupt listener for the audio session (taken from LibPd)
void audioSessionInterruptListener(void *inClientData, UInt32 inInterruption) {
    AudioController *controller = (AudioController *)inClientData;
    switch (inInterruption) {
        case kAudioSessionBeginInterruption: {
            // when the interruption begins, suspend audio playback
            NSLog(@"AudioSession === kAudioSessionBeginInterruption");
            [controller pause];
            break;
        }
        case kAudioSessionEndInterruption: {
            // when the interruption ends, resume audio playback
            NSLog(@"AudioSession === kAudioSessionEndInterruption");
            [controller play];
            break;
        }
        default: {
            break;
        }
    }
}

@interface AudioController(PrivateMethods)
/*!
 @method initAudioSession
 @abstract Intializes the Audio Session for CoreAudio
 */
- (void)initAudioSession;

/*!
 @method initAudioUnit
 @abstract Intializes an Audio Unit for CoreAudio. This Audio Unit contains a reference to the renderCallback-function
 */
- (void)initAudioUnit;
@end

@implementation AudioController

@synthesize masterVolume;
@synthesize balance;

-(id) init {
    if ((self = [super init])) {
        sinPhase = 0.0f;
        sinFreq = kBaseFreq;
        masterVolume = 0.0f;
        [self setBalance:0.0f];
        
        // set up stk stuff
        Stk::setSampleRate(kSampleRate);
        
        // Frames holds the values that will be written to the buffer in renderCallback()
        frames = new StkFrames(kBufferSize, kNumOuputChannels);
        
        // Sine wave generator
        sine = new SineWave;
        sine->setFrequency(kBaseFreq);
        
        // Noise generator
        noise = new Noise;
        noise->setSeed();
        
        // start up
        [self initAudioSession];
        [self initAudioUnit];
    }
    
    return self;
}

- (void)play {
    AudioOutputUnitStart(audioUnit);
    NSLog(@"AudioSession === starting audio unit.");
}

- (void)pause {
    AudioOutputUnitStop(audioUnit);
    NSLog(@"AudioSession === stopping audio unit.");
}

- (void)setBalance:(float)v {
    balance = v;
    channelVol[0] = (1.0f - v) / 2.0f;
    channelVol[1] = (1.0f + v) / 2.0f;
}

// Clean up memory
- (void)dealloc {
    [self pause];
    
    delete frames;
    delete sine;
    delete noise;
    
    free(audioUnit);
    audioUnit = nil;
    
    [super dealloc];
}

// partly taken from LibPd:
- (void)initAudioSession {
    /*** Create AudioSession interface to Core Audio === ***/
    
    // initialise the audio session
    AudioSessionInitialize(NULL, NULL, audioSessionInterruptListener, self);
    
    AudioSessionSetProperty(kAudioSessionProperty_AudioCategory, sizeof(kAudioCategory), &kAudioCategory);
    
    // set the sample rate of the session
    AudioSessionSetProperty(kAudioSessionProperty_PreferredHardwareSampleRate, sizeof(kSampleRate), &kSampleRate);
    NSLog(@"AudioSession === setting PreferredHardwareSampleRate to %.0fHz.", kSampleRate);
    
    // set buffer size
    Float32 bufferSize = kBufferSize; // requested buffer size
    Float32 bufferDuration = (bufferSize + 0.5) / kSampleRate; // buffer duration in seconds - add 0.5 due to differences in armv6 and armv7 fp
    AudioSessionSetProperty(kAudioSessionProperty_PreferredHardwareIOBufferDuration, 
                            sizeof(bufferDuration), &bufferDuration);
    NSLog(@"AudioSession === setting PreferredHardwareIOBufferDuration to %3.2fms.", bufferDuration*1000.0);
    
    // NOTE: note that round-off errors make it hard to determine whether the requested buffersize
    // was granted. we just assume that it was and carry on.
    
    AudioSessionSetActive(true);
    NSLog(@"AudioSession === starting Audio Session.");
    
    // print value of properties to check that everything was set properly
    Float64 audioSessionProperty64 = 0;
    Float32 audioSessionProperty32 = 0;
    UInt32 audioSessionPropertySize64 = sizeof(audioSessionProperty64);
    UInt32 audioSessionPropertySize32 = sizeof(audioSessionProperty32);
    AudioSessionGetProperty(kAudioSessionProperty_CurrentHardwareSampleRate, 
                            &audioSessionPropertySize64, &audioSessionProperty64);
    NSLog(@"AudioSession === CurrentHardwareSampleRate: %.0fHz", audioSessionProperty64);
//    sampleRate = audioSessionProperty64;
    
    AudioSessionGetProperty(kAudioSessionProperty_CurrentHardwareIOBufferDuration, 
                            &audioSessionPropertySize32, &audioSessionProperty32);
    int blockSize = lrint(audioSessionProperty32 * audioSessionProperty64);
    NSLog(@"AudioSession === CurrentHardwareIOBufferDuration: %3.2fms", audioSessionProperty32*1000.0f);
    NSLog(@"AudioSession === block size: %i", blockSize);
}

// partly taken from LibPd:
- (void)initAudioUnit {
    // http://developer.apple.com/iphone/library/documentation/Audio/Conceptual/AudioUnitLoadingGuide_iPhoneOS/AccessingAudioUnits/LoadingIndividualAudioUnits.html#//apple_ref/doc/uid/TP40008781-CH103-SW11
    
    // create an AudioComponentDescription describing a RemoteIO audio unit
    // such a component provides an interface from microphone to speaker
    AudioComponentDescription auDescription;
    auDescription.componentType = kAudioUnitType_Output;
    auDescription.componentSubType = kAudioUnitSubType_RemoteIO;
    auDescription.componentManufacturer = kAudioUnitManufacturer_Apple;
    auDescription.componentFlags = 0;
    auDescription.componentFlagsMask = 0;
    
    // find an audio component fitting the given description
    AudioComponent foundComponent = AudioComponentFindNext(NULL, &auDescription);
    
    // create a new audio unit instance
    AudioComponentInstanceNew(foundComponent, &audioUnit);
    
    // connect the AU to hardware input and output
    OSStatus err = 0; // http://developer.apple.com/iphone/library/documentation/AudioUnit/Reference/AUComponentServicesReference/Reference/reference.html
    UInt32 doSetProperty = 1;
    AudioUnitElement inputBus = 1;
    AudioUnitElement outputBus = 0;
    // connect the AU to the microphone 
    if (kNumInputChannels > 0) {
        AudioUnitSetProperty(audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Input, 
                             inputBus, &doSetProperty, sizeof(doSetProperty));
    }
    
    // connect the AU to the soundout
    AudioUnitSetProperty(audioUnit, kAudioOutputUnitProperty_EnableIO, kAudioUnitScope_Output, 
                         outputBus, &doSetProperty, sizeof(doSetProperty));
    
    // set the sample rate on the input and output busses
    AudioUnitSetProperty(audioUnit, kAudioUnitProperty_SampleRate, kAudioUnitScope_Input, outputBus, 
                         &kSampleRate, sizeof(kSampleRate));
    
    if (kNumInputChannels > 0) {
        AudioUnitSetProperty(audioUnit, kAudioUnitProperty_SampleRate, kAudioUnitScope_Output, inputBus, 
                             &kSampleRate, sizeof(kSampleRate));
    }
    
    // request the audio data stream format for input and output
    // NOTE: this really is a request. The system will set the stream to whatever it damn well pleases.
    // The settings here are what are known to work: 16-bit mono (interleaved) @ 22050
    // and thus no rigorous checking is done in order to ensure that the request stream format
    // is actually being used. It would be nice to be able to use format kAudioFormatFlagsNativeFloatPacked
    // which would allow us to avoid converting between float and int sample type manually.
    if (kNumInputChannels > 0) {
        AudioStreamBasicDescription toneStreamFormatInput;
        memset (&toneStreamFormatInput, 0, sizeof (toneStreamFormatInput)); // clear all fields
        toneStreamFormatInput.mSampleRate       = kSampleRate;
        toneStreamFormatInput.mFormatID         = kAudioFormatLinearPCM;
        toneStreamFormatInput.mFormatFlags      = kAudioFormatFlagsCanonical;
        toneStreamFormatInput.mBytesPerPacket   = 2 * kNumInputChannels;
        toneStreamFormatInput.mFramesPerPacket  = 1;
        toneStreamFormatInput.mBytesPerFrame    = 2 * kNumInputChannels;
        toneStreamFormatInput.mChannelsPerFrame = kNumInputChannels;
        toneStreamFormatInput.mBitsPerChannel   = 16;
        // apply the audio data stream format to bus 0 of the input scope of the Remote I/O AU. This is
        // actually the OUTPUT to the system.
        err = AudioUnitSetProperty(audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input, outputBus, 
                                   &toneStreamFormatInput, sizeof(toneStreamFormatInput));
    }
    
    // set audio output format to 16-bit stereo
    AudioStreamBasicDescription toneStreamFormatOutput;
    memset (&toneStreamFormatOutput, 0, sizeof(toneStreamFormatOutput));
    toneStreamFormatOutput.mSampleRate       = kSampleRate;
    toneStreamFormatOutput.mFormatID         = kAudioFormatLinearPCM;
    toneStreamFormatOutput.mFormatFlags      = kAudioFormatFlagsCanonical;
    toneStreamFormatOutput.mBytesPerPacket   = 2 * kNumOuputChannels;
    toneStreamFormatOutput.mFramesPerPacket  = 1;
    toneStreamFormatOutput.mBytesPerFrame    = 2 * kNumOuputChannels;
    toneStreamFormatOutput.mChannelsPerFrame = kNumOuputChannels;
    toneStreamFormatOutput.mBitsPerChannel   = 16;
    
    // apply the audio data stream format to bus 1 of the output scope of the Remote I/O AU. This is
    // actually the INPUT to the system.
    AudioUnitSetProperty(audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Output, inputBus, 
                         &toneStreamFormatOutput, sizeof(toneStreamFormatOutput));
    
    // register the render callback. This is the function that the audio unit calls when it needs audio
    // the callback function (renderCallback()) is defined at the top of the page.
    AURenderCallbackStruct renderCallbackStruct;
    renderCallbackStruct.inputProc = renderCallback;
    renderCallbackStruct.inputProcRefCon = self; // this is an optional data pointer
    // pass the AudioController object
    
    AudioUnitSetProperty(audioUnit, kAudioUnitProperty_SetRenderCallback, kAudioUnitScope_Input, 
                         outputBus, &renderCallbackStruct, sizeof(renderCallbackStruct));
    
    // disable buffer allocation on output (necessary?)
    // http://developer.apple.com/iphone/library/documentation/Audio/Conceptual/AudioUnitLoadingGuide_iPhoneOS/AccessingAudioUnits/LoadingIndividualAudioUnits.html#//apple_ref/doc/uid/TP40008781-CH103-SW19
    doSetProperty = 0;
    AudioUnitSetProperty(audioUnit, kAudioUnitProperty_ShouldAllocateBuffer, kAudioUnitScope_Output, 
                         outputBus, &doSetProperty, sizeof(doSetProperty));
    
    // finally, initialise the audio unit. It is ready to go.
    AudioUnitInitialize(audioUnit);
    
    // ensure that all parameters and settings have been successfully applied
    UInt32 toneStreamFormatSize = sizeof(AudioStreamBasicDescription);
    
    if (kNumInputChannels > 0) {
        AudioStreamBasicDescription toneStreamFormatInputTest;
        memset(&toneStreamFormatInputTest, 0, sizeof(toneStreamFormatInputTest));
        AudioUnitGetProperty(audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Input,
                             outputBus, &toneStreamFormatInputTest, &toneStreamFormatSize);
        NSLog(@"=== input stream format:");
        NSLog(@"  mSampleRate: %.0fHz", toneStreamFormatInputTest.mSampleRate);
        NSLog(@"  mFormatID: %lu", toneStreamFormatInputTest.mFormatID);
        NSLog(@"  mFormatFlags: %lu", toneStreamFormatInputTest.mFormatFlags);
        NSLog(@"  mBytesPerPacket: %lu", toneStreamFormatInputTest.mBytesPerPacket);
        NSLog(@"  mFramesPerPacket: %lu", toneStreamFormatInputTest.mFramesPerPacket);
        NSLog(@"  mBytesPerFrame: %lu", toneStreamFormatInputTest.mBytesPerFrame);
        NSLog(@"  mChannelsPerFrame: %lu", toneStreamFormatInputTest.mChannelsPerFrame);
        NSLog(@"  mBitsPerChannel: %lu", toneStreamFormatInputTest.mBitsPerChannel);
    }
    
    
    AudioStreamBasicDescription toneStreamFormatOutputTest;
    memset (&toneStreamFormatOutputTest, 0, sizeof(toneStreamFormatOutputTest));
    AudioUnitGetProperty(audioUnit, kAudioUnitProperty_StreamFormat, kAudioUnitScope_Output,
                         inputBus, &toneStreamFormatOutputTest, &toneStreamFormatSize);
    NSLog(@"=== output stream format:");
    NSLog(@"  mSampleRate: %.0fHz", toneStreamFormatOutputTest.mSampleRate);
    NSLog(@"  mFormatID: %lu", toneStreamFormatOutputTest.mFormatID);
    NSLog(@"  mFormatFlags: %lu", toneStreamFormatOutputTest.mFormatFlags);
    NSLog(@"  mBytesPerPacket: %lu", toneStreamFormatOutputTest.mBytesPerPacket);
    NSLog(@"  mFramesPerPacket: %lu", toneStreamFormatOutputTest.mFramesPerPacket);
    NSLog(@"  mBytesPerFrame: %lu", toneStreamFormatOutputTest.mBytesPerFrame);
    NSLog(@"  mChannelsPerFrame: %lu", toneStreamFormatOutputTest.mChannelsPerFrame);
    NSLog(@"  mBitsPerChannel: %lu", toneStreamFormatOutputTest.mBitsPerChannel);
    
    Float64 auSampleRate = 0.0;
    UInt32 sampleRateSize = sizeof(auSampleRate);
    AudioUnitGetProperty(audioUnit, kAudioUnitProperty_SampleRate, kAudioUnitScope_Input, outputBus, &auSampleRate, &sampleRateSize);
    NSLog(@"=== input sample rate: %.0fHz", auSampleRate);
    
    AudioUnitGetProperty(audioUnit, kAudioUnitProperty_SampleRate, kAudioUnitScope_Output, inputBus, &auSampleRate, &sampleRateSize);
    NSLog(@"=== output sample rate: %.0fHz", auSampleRate);
}

@end
