//
//  SeqView.m
//  PdAudio
//
//  Created by Markus Konrad on 19.12.10.
//  Copyright 2010 __MyCompanyName__. All rights reserved.
//

#import "SeqView.h"

#import "Pentatonics.h"

@implementation SeqView

@synthesize buttons;

- (id)initWithFrame:(CGRect)frame andParentViewController:(PdSoundEngineViewController *)parent {
    
    self = [super initWithFrame:frame];
    if (self) {
        // set defaults:
        trackNoteDur = 1.0f;
        
        // weak refs:
        parentViewController = parent;
        soundEngine = parent.soundEngine;
        
        // create a track in the SoundEngine for this SeqView:
        firstNoteSet = NO;
        trackId = [soundEngine.sequencer addTrackWithView:self andInstrument:1 andVolume:1.0f andLength:INITIAL_SEQUENCER_BARS];
        
        // create a dictionary for mapping button-tags to note-ids
        noteIds = [[NSMutableDictionary alloc] initWithCapacity:1];
        
        //self.backgroundColor = [UIColor orangeColor];
        
        // create step-sequencer-like button rows
        buttons = [[NSMutableArray alloc] initWithCapacity:INITIAL_SEQUENCER_TRACKS];
        [self _createInitialButtons];
        [self _setButtonTags];
        
        
        // create "change instrument" button
        changeInstrButton = [UIButton buttonWithType:UIButtonTypeRoundedRect];
        changeInstrButton.frame = CGRectMake(0,
                                        ([buttons count] + 1) * (SEQ_VIEW_BUTTON_H + SEQ_VIEW_BUTTON_BORDER),
                                        SEQ_VIEW_BUTTON_W,
                                        SEQ_VIEW_BUTTON_H);
        [changeInstrButton setTitle:@"1" forState:UIControlStateNormal];
        [changeInstrButton addTarget:self action:@selector(changeInstrAction:) forControlEvents:UIControlEventTouchUpInside];
        
        [self addSubview:changeInstrButton];
        
        // create track volume slider
        trackVolSlider = [[UISlider alloc] initWithFrame:CGRectMake(SEQ_VIEW_BUTTON_W + SEQ_VIEW_BUTTON_BORDER,
                                                                     ([buttons count] + 1) * (SEQ_VIEW_BUTTON_H + SEQ_VIEW_BUTTON_BORDER),
                                                                     200,
                                                                     SEQ_VIEW_BUTTON_H)];
        trackVolSlider.minimumValue = 0.0f;
        trackVolSlider.maximumValue = 1.0f;
        trackVolSlider.value = 1.0f;
        [trackVolSlider addTarget:self action:@selector(trackVolChangedAction:) forControlEvents:UIControlEventValueChanged];
        [self addSubview:trackVolSlider];
        
        // create track note duration slider
        trackNoteDurSlider = [[UISlider alloc] initWithFrame:CGRectMake(SEQ_VIEW_BUTTON_W + 200 + 2 * SEQ_VIEW_BUTTON_BORDER,
                                                                    ([buttons count] + 1) * (SEQ_VIEW_BUTTON_H + SEQ_VIEW_BUTTON_BORDER),
                                                                    200,
                                                                    SEQ_VIEW_BUTTON_H)];
        trackNoteDurSlider.minimumValue = 0.1f;
        trackNoteDurSlider.maximumValue = 4.0f;
        trackNoteDurSlider.value = trackNoteDur;
        [trackNoteDurSlider addTarget:self action:@selector(trackNoteDurChangedAction:) forControlEvents:UIControlEventValueChanged];
        [self addSubview:trackNoteDurSlider];
        
        // create "add column" button
        addColumnButton = [UIButton buttonWithType:UIButtonTypeRoundedRect];
        addColumnButton.frame = CGRectMake(self.frame.size.width - 2 * (SEQ_VIEW_BUTTON_W + SEQ_VIEW_BUTTON_BORDER),
                                        0,
                                        SEQ_VIEW_BUTTON_W,
                                        SEQ_VIEW_BUTTON_H);
        [addColumnButton setTitle:@"+" forState:UIControlStateNormal];
        [addColumnButton addTarget:self action:@selector(addButtonColumnAction:) forControlEvents:UIControlEventTouchUpInside];
        
        [self addSubview:addColumnButton];
        
        // create "remove column" button
        removeColumnButton = [UIButton buttonWithType:UIButtonTypeRoundedRect];
        removeColumnButton.frame = CGRectMake(self.frame.size.width - 1 * (SEQ_VIEW_BUTTON_W + SEQ_VIEW_BUTTON_BORDER),
                                           0,
                                           SEQ_VIEW_BUTTON_W,
                                           SEQ_VIEW_BUTTON_H);
        [removeColumnButton setTitle:@"-" forState:UIControlStateNormal];
        [removeColumnButton addTarget:self action:@selector(removeButtonColumnAction:) forControlEvents:UIControlEventTouchUpInside];
        
        [self addSubview:removeColumnButton];
    }
    return self;
}


- (void)dealloc {
    parentViewController = nil;
    soundEngine = nil;
    
    [noteIds release];
    
    [soundEngine.sequencer deleteTrack:trackId];
    
    [trackVolSlider release];
    [trackNoteDurSlider release];
    [buttons release];
    
    [super dealloc];
}

- (void)addButtonRow {    
    // create empty button row array
    NSMutableArray *buttonArray = [[NSMutableArray alloc] initWithCapacity:INITIAL_SEQUENCER_BARS];
    [buttons addObject:buttonArray];
    [buttonArray release];
    int curRowNum = [buttons count] - 1;
        
    // add step seq. buttons to the button row array
    int columnCount = INITIAL_SEQUENCER_BARS;
    if ([buttons count] > 0 && [[buttons objectAtIndex:0] count] > 0) {
        columnCount = [[buttons objectAtIndex:0] count];
    }
    
    for (int cell = 0; cell < columnCount; cell++) {
        [self addButtonToRow:curRowNum];
    }
}

- (void)addButtonColumn {
    for (int row = 0; row < [buttons count]; row++) {
        [self addButtonToRow:row];
    }
    
    // add bar to the sequencer
    [soundEngine.sequencer addBarToTrack:trackId];
}

- (void)addButtonToRow:(int)rowNum {
    NSMutableArray *buttonArray = [buttons objectAtIndex:rowNum];
    
    UIButton *btn = [UIButton buttonWithType:UIButtonTypeRoundedRect];
    btn.frame = CGRectMake(SEQ_VIEW_ROW_BORDER_LEFT + [buttonArray count] * (SEQ_VIEW_BUTTON_W + SEQ_VIEW_BUTTON_BORDER),
                           rowNum * (SEQ_VIEW_BUTTON_H + SEQ_VIEW_BUTTON_BORDER),
                           SEQ_VIEW_BUTTON_W,
                           SEQ_VIEW_BUTTON_H);
    [btn setTitle:@" " forState:UIControlStateNormal];
    [btn addTarget:self action:@selector(toggleCell:) forControlEvents:UIControlEventTouchUpInside];
    
    [buttonArray addObject:btn];
    [self addSubview:btn];
}

- (void)removeButtonColumn {
    UIButton *btn;
    if ([buttons count] > 0 && [[buttons objectAtIndex:0] count] > 0) {
        for (int row = 0; row < [buttons count]; row++) {
            btn = [[buttons objectAtIndex:row] lastObject];
            [noteIds removeObjectForKey:[NSNumber numberWithInt:btn.tag]];
            [btn removeFromSuperview];
            [[buttons objectAtIndex:row] removeLastObject];
        }
    }
    
    // remove bar from the sequencer
    [soundEngine.sequencer deleteBarFromTrack:trackId];
}

#pragma mark action messages

- (void)addButtonColumnAction:(id)sender {
    if ([[buttons objectAtIndex:0] count] < MAX_SEQUENCER_BARS) {
        [self addButtonColumn];
        [self _setButtonTags];
    }
}
- (void)removeButtonColumnAction:(id)sender {
    if ([[buttons objectAtIndex:0] count] > 1) {
        [self removeButtonColumn];
        [self _setButtonTags];
    }
}

- (void)trackVolChangedAction:(id)sender {
    [soundEngine.sequencer setAttrib:kTrackAttribVolume value:trackVolSlider.value forTrack:trackId];
}

- (void)trackNoteDurChangedAction:(id)sender {
    trackNoteDur = trackNoteDurSlider.value;
}

- (void)toggleCell:(id)sender {
    UIButton *btn = (UIButton *)sender;
    
    int columnCount = [[buttons objectAtIndex:0] count];
    int seqRow = btn.tag / columnCount;
    int seqCell = btn.tag % columnCount; 
    
    NSLog(@"Clicked %d, %d", seqRow, seqCell);
    
    NSNumber *noteIdDictKey = [NSNumber numberWithInt:btn.tag];
    
    if (btn.titleLabel.text == @" ") {
        [btn setTitle:@"X" forState:UIControlStateNormal];
        
        MIDINote *note = [[MIDINote alloc] initWithPitch:(36 + [Pentatonics pentatonicPitchForStep:seqRow])
                                             andVelocity:90.0f
                                             andDuration:2.0f * trackNoteDur];
        int noteId = [soundEngine.sequencer insertNote:note toTrack:trackId atBar:seqCell];
        [noteIds setObject:[NSNumber numberWithInt:noteId] forKey:noteIdDictKey];
        
        [note release];
    } else {
        [btn setTitle:@" " forState:UIControlStateNormal];
        [soundEngine.sequencer deleteNote:[(NSNumber *)[noteIds objectForKey:noteIdDictKey] intValue]
                                  fromBar:seqCell
                                  ofTrack:trackId];
        [noteIds removeObjectForKey:noteIdDictKey];
    }
    
    if (firstNoteSet == NO) {
        [soundEngine.sequencer playTrack:trackId];
        
        [parentViewController.nextSeqView setHidden:NO];
        
        firstNoteSet = YES;
    }
}

- (void)changeInstrAction:(id)sender {
    int curInstr = (int)[soundEngine.sequencer getAttrib:kTrackAttribInstrument forTrack:trackId];
    curInstr++;
    if (curInstr > NUM_INSTRUMENTS) {
        curInstr = 1;
    }
    
    [soundEngine.sequencer setAttrib:kTrackAttribInstrument value:(float)curInstr forTrack:trackId];
    [changeInstrButton setTitle:[NSString stringWithFormat:@"%d", curInstr] forState:UIControlStateNormal];
}

#pragma mark SequencerViewDelegate messages

-(void)playsNote:(int)noteId atBar:(int)bar {
    NSLog(@" > Played note#%d at bar %d", noteId, bar);
    
    UIButton *playedBtn = [self _buttonForNoteId:noteId atBar:bar];
    [playedBtn setHighlighted:YES];
}

-(void)stopsNote:(int)noteId atBar:(int)bar {
    NSLog(@" > Stopped note#%d at bar %d", noteId, bar);    
    
    UIButton *playedBtn = [self _buttonForNoteId:noteId atBar:bar];
    [playedBtn setHighlighted:NO];
}


#pragma mark private messages

- (void)_createInitialButtons {    
    for (int row = 0; row < INITIAL_SEQUENCER_TRACKS; row++) {
        [self addButtonRow];
    }
}

- (void)_moveElement:(UIView *)view byX:(int)x andY:(int)y {    
    CGRect newFrame = CGRectMake(view.frame.origin.x + x,
                                 view.frame.origin.y + y,
                                 view.frame.size.width,
                                 view.frame.size.height);
    [view setFrame:newFrame];
}

- (void)_setButtonTags {
    if ([buttons count] <= 0)
        return;
    
    int btnNum = 0;
    
    NSMutableDictionary *newNoteIds = [[NSMutableDictionary alloc] initWithCapacity:[noteIds count]];
    
    @synchronized(self) {   //this is critical code when _buttonForNoteId:atBar: is called at the same time
        for (NSMutableArray *buttonArray in buttons) {
            for (UIButton *btn in buttonArray) {
                NSNumber *oldBtnTag = [NSNumber numberWithInt:btn.tag];
                NSNumber *noteId = [[noteIds objectForKey:oldBtnTag] retain];   // get noteId for old btn.tag
                
                if (noteId != nil) {
                    [noteIds removeObjectForKey:oldBtnTag];   // delete noteId for old btn.tag
                    //NSLog(@"Removed noteId#%d for old button %d", [noteId intValue], [oldBtnTag intValue]);
                }
                
                btn.tag = btnNum;
                
                if (noteId != nil) {
                    [newNoteIds setObject:noteId forKey:[NSNumber numberWithInt:btn.tag]];  // set noteId for new btn.tag
                    //NSLog(@"Added noteId#%d for new button %d", [noteId intValue], btn.tag);
                }
                
                [noteId release];
                
                btnNum++;
            }
        }
    }
    
    [noteIds release];
    noteIds = newNoteIds;
}

- (UIButton *)_buttonForNoteId:(int)noteId atBar:(int)bar {
    UIButton *foundBtn = nil;
    
    @synchronized(self) {   //this is critical code when _setButtonTags is called at the same time
        // get all bars that have notes with that noteId (a noteId is unique only in each bar)
        NSArray *btnTags = [noteIds allKeysForObject:[NSNumber numberWithInt:noteId]];
        
        int numBars = [[buttons objectAtIndex:0] count];
        int btnTag, barOfButton, rowOfButton;
        for (NSNumber *btnTagNumber in btnTags) {   // for each of the found button tags look if its the one in the bar we want
            btnTag = [btnTagNumber intValue];
            rowOfButton = btnTag / numBars;
            barOfButton = btnTag % numBars;
            
            if (barOfButton == bar) {
                foundBtn = [[buttons objectAtIndex:rowOfButton] objectAtIndex:barOfButton];
                break;
            }
        }
    }
    
    return foundBtn;
}

@end
