//
//  Sequencer.m
//  AudioSeq
//
//  Created by Markus Konrad on 14.12.10.
//  Copyright 2010 __MyCompanyName__. All rights reserved.
//

#import "Sequencer.h"

@implementation Sequencer

int seqNoteIdCounter = 0;

@synthesize speed;

# pragma mark NSObject messages

- (id)initWithMidiDelegate:(id)pMidiDelegate {
    if  (self=[super init]) {
        midiDelegate = pMidiDelegate;
		tracks = [[NSMutableArray alloc] initWithCapacity:1];
		trackAttrib = [[NSMutableArray alloc] initWithCapacity:1];
        trackBarCounters = [[NSMutableArray alloc] initWithCapacity:1];
        viewDelegates = [[NSMutableArray alloc] initWithCapacity:1];
        speed = 60.0f;
	}
    
    return self;
}

-(void)dealloc {
    [midiDelegate release];
    [viewDelegates release];
    
    [tracks release];
    [trackBarCounters release];
    [trackAttrib release];
    
    [super dealloc];
}

# pragma mark public messages

- (int)addTrackWithView:(id)viewDelegate andInstrument:(int)instrument andVolume:(float)volume andLength:(int)length {
    // create array for notes
    NSMutableArray *trackArray = [NSMutableArray arrayWithCapacity:length];    // each track consists of <length> bars, where each bar holds an array of MIDINotes
    for (int i = 0; i < length; i++) {
        [trackArray addObject:[NSMutableDictionary dictionaryWithCapacity:0]];   // initialize each bar with empty dictionary
    }
	[tracks addObject:trackArray]; // add the array to the track-array
    
    // create array for attributes
    NSMutableArray* attribs = [NSMutableArray arrayWithObjects:
                               [NSNumber numberWithInt:instrument],
                               [NSNumber numberWithFloat: volume],
                               [NSNumber numberWithFloat: length],
                               nil];
	[trackAttrib addObject: attribs];
    
    // initialize track bar counter to 0 for this track
    [trackBarCounters addObject:[NSNumber numberWithInt:0]];
    
    // add the view delegate
    [viewDelegates addObject:viewDelegate];
    
	return tracks.count - 1 ;
}

/*
 * adding a particular note to a given track at a given position
 * @return index of the input note
 */
- (int)insertNote:(MIDINote *)note toTrack:(int)track atBar:(int)bar {
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");    
    NSAssert(bar >= 0, @"Invalid bar");
    NSAssert(bar < (int)[self getAttrib:kTrackAttribLength forTrack:track], @"Invalid bar");

    NSMutableDictionary *trackBar = [[tracks objectAtIndex:track] objectAtIndex:bar];    // contains all notes for this track
    seqNoteIdCounter++;
    [trackBar setObject:note forKey:[NSNumber numberWithInt:seqNoteIdCounter]]; 
    
    return seqNoteIdCounter;
}

/*
 *	calls the playNotes method of a given track strored inside of the "tracks" array
 *	
 */
- (void)playTrack:(int)track {    
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    
    //NSLog(@"Playing track #%d", track);
    
	playing = YES;
	[self _playNotesOfTrack:[NSNumber numberWithInt:track]];
}

- (void)setAttrib:(int)attrIndex value:(float)v forTrack:(int)track {
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    
    NSMutableArray *attribArray = [trackAttrib objectAtIndex:track];
    
    NSAssert(attrIndex >= 0, @"Invalid attribute key");
    NSAssert(attrIndex < [attribArray count], @"Invalid attribute key");
    
    [attribArray replaceObjectAtIndex:attrIndex withObject:[NSNumber numberWithFloat:v]];
}
       
- (float)getAttrib:(int)attrIndex forTrack:(int)track {
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    
    NSMutableArray *attribArray = [trackAttrib objectAtIndex:track];
    
    NSAssert(attrIndex >= 0, @"Invalid attribute key");
    NSAssert(attrIndex < [attribArray count], @"Invalid attribute key");
    
    
    return [(NSNumber *)[attribArray objectAtIndex:attrIndex] floatValue];
}

/*
 * breaks the loop
 *
 */
- (void)pauseTrack:(int)track{
	playing = NO;
}

/*
 * deletes a particular track of the sequenzer
 *
 */
- (void)deleteTrack:(int)track{
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    NSAssert(track < [trackAttrib count], @"Invalid trackId");
    NSAssert(track < [trackBarCounters count], @"Invalid trackId");
    NSAssert(track < [viewDelegates count], @"Invalid trackId");
    
	[tracks removeObjectAtIndex:track];
    [trackAttrib removeObjectAtIndex:track];
    [trackBarCounters removeObjectAtIndex:track];
    [viewDelegates removeObjectAtIndex:track];
}

- (void)deleteNote:(int)noteId fromBar:(int)bar ofTrack:(int)track {
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    NSAssert(bar >= 0, @"Invalid bar");
    NSAssert(bar < (int)[self getAttrib:kTrackAttribLength forTrack:track], @"Invalid bar");
    
    NSMutableDictionary *trackBar = [[tracks objectAtIndex:track] objectAtIndex:bar];    // contains all notes for this track
    
    NSAssert(noteId > 0, @"Invalid noteId");
    NSAssert(noteId <= seqNoteIdCounter, @"Invalid noteId");
    
    [trackBar removeObjectForKey:[NSNumber numberWithInt:noteId]];
}

- (void)addBarToTrack:(int)track {
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    NSAssert(track < [trackAttrib count], @"Invalid trackId");
    
    // add empty array for new notes in this new bar
    [[tracks objectAtIndex:track] addObject:[NSMutableDictionary dictionaryWithCapacity:0]];
    
    // increase track length
    [self setAttrib:kTrackAttribLength
              value:([self getAttrib:kTrackAttribLength forTrack:track] + 1.0f)
           forTrack:track];
}

- (void)deleteBarFromTrack:(int)track {
    NSAssert(track >= 0, @"Invalid trackId");
    NSAssert(track < [tracks count], @"Invalid trackId");
    NSAssert(track < [trackAttrib count], @"Invalid trackId");
    
    // remove last bar -> also destroys the notes in it
    [[tracks objectAtIndex:track] removeLastObject];
    
    // decrease track length
    [self setAttrib:kTrackAttribLength
              value:([self getAttrib:kTrackAttribLength forTrack:track] - 1.0f)
           forTrack:track];
}

#pragma mark private messages

/*
 * plays stored notes inside of one given track.
 *
 */
- (void)_playNotesOfTrack:(NSNumber *)track {
	int trackId = [track intValue];
    
    NSAssert(trackId >= 0, @"Invalid trackId");
    NSAssert(trackId < [tracks count], @"Invalid trackId");
    NSAssert(trackId < [trackBarCounters count], @"Invalid trackId");
    NSAssert(trackId < [trackAttrib count], @"Invalid trackId");
    NSAssert(trackId < [viewDelegates count], @"Invalid trackId");
    
    // get array of bars of this track
    NSMutableArray *barArray = [tracks objectAtIndex:trackId];
    if (!playing || [barArray count] <= 0) {
        return; // stop playing
    }
    
    // get seconds per beat
    double beatTime = (double)(60.0 / speed);
    
    // get current bar
    int curBar = [((NSNumber *)[trackBarCounters objectAtIndex:trackId]) intValue];
    
    if (curBar < [barArray count]) {
        // send playing bar event to viewdelegate
        id viewDelegate = [viewDelegates objectAtIndex:trackId];
        if ([viewDelegate respondsToSelector:@selector(playsBar:)]) {
            [viewDelegate playsBar:curBar];
        }
        
        // get all notes of this bar in this track and play them
        int instrNum = [self getAttrib:kTrackAttribInstrument forTrack:trackId];
        float trackVol = [self getAttrib:kTrackAttribVolume forTrack:trackId];
        
        [[barArray objectAtIndex:curBar] enumerateKeysAndObjectsUsingBlock: ^(id noteId, id note, BOOL *stop) {
            // send play note message to the delegates
            [midiDelegate playsNote:note withInstrument:instrNum atVolume:trackVol];
            [viewDelegate playsNote:[noteId intValue] atBar:curBar];
            
            // schedule stop-note event
            [self performSelector:@selector(_stopNote:)
                       withObject:[NSArray arrayWithObjects:
                                   note,
                                   noteId,
                                   [NSNumber numberWithInt:instrNum],
                                   [trackBarCounters objectAtIndex:trackId],
                                   track,
                                   nil]
                       afterDelay:[(MIDINote *)note duration] * beatTime];
        }];
                
        // increment current bar
        curBar++;
        curBar = curBar % (int)[self getAttrib:kTrackAttribLength forTrack:trackId]; // lets keep curBar between 0 and track-length
    } else {
        curBar = 0;
    }

    // schedule playing of the next bar
    [self performSelector:@selector(_playNotesOfTrack:)
               withObject:track
               afterDelay:beatTime];
    
    [trackBarCounters replaceObjectAtIndex:trackId
                                withObject:[NSNumber numberWithInt:curBar]];
}


/*
 * stops particular notes stored in a row -> sending note off signal to the world outside
 *
 */
- (void)_stopNote:(NSArray *)paramArray {
    int trackId = [(NSNumber *)([paramArray objectAtIndex:4]) intValue];
    NSAssert(trackId >= 0, @"Invalid trackId");
    NSAssert(trackId < [viewDelegates count], @"Invalid trackId");
    
	[midiDelegate stopsNote:(MIDINote *)([paramArray objectAtIndex:0])
               ofInstrument:[(NSNumber *)([paramArray objectAtIndex:2]) intValue]];
    
    [[viewDelegates objectAtIndex:trackId] stopsNote:[(NSNumber *)([paramArray objectAtIndex:1]) intValue]
                                               atBar:[(NSNumber *)([paramArray objectAtIndex:3]) intValue]];
}


@end
