#include "Client.hpp"
#include "../net/NetworkMessages.hpp"

CClient::CClient(CGame* pGame, unsigned int nRefreshInterval,
		unsigned int nSendInterval, unsigned int nMessageQueueLength)
			: BaseThread()
{
	m_nRefreshInterval = nRefreshInterval;
	m_nSendInterval = nSendInterval;
	m_nMessageQueueLength = nMessageQueueLength;
	m_nLastRefresh = 0;
	m_nLastSend = 0;

	m_pGame = pGame;
	//m_pGameInfo = NULL;
	memset(&m_sGameInfo, 0, sizeof(m_sGameInfo));
	m_pLocalGameController = NULL;
	m_pMutex = SDL_CreateMutex();

	m_pRemoteGames = NULL;
	m_pRemotePlayer = NULL;
	m_pRemoteMaps = NULL;
	m_szCurrentMap = NULL;
	m_pLocalPlayer = NULL;

	m_bBlock = false;
	m_bInGame = false;

	m_sSocket = NULL;

	m_nAuthToken = 0;
	m_nLocPlayerID = 0;
	m_nCurrGameID = -1;
	m_nPlayerStartPos = -1;

	m_nNextLevelStartTime = 0;
	m_nLastServerTimeRecv = 0;
	m_nServerTime = 0;
}

CClient::~CClient()
{
	Terminate();

	FREEMEM(m_pRemoteGames);
	FREEMEM(m_pRemotePlayer);
	FREEMEM(m_pRemoteMaps);

	if (m_sSocket != NULL)
	{
		SDLNet_TCP_Close(m_sSocket);
		DEBUG_PRINTL("Disconnected from server");
	}

	SDLNet_Quit();
	SDL_DestroyMutex(m_pMutex);
}

bool CClient::connect(const char* szServer, int nPort)
{
	if (SDLNet_Init() < 0)
	{
		DEBUG_PRINTL("Couldn't init SDL_Net: " << SDLNet_GetError());
		return false;
	}

	IPaddress addr;
	if (SDLNet_ResolveHost(&addr, szServer, nPort) < 0)
	{
		DEBUG_PRINTL("Couldn't connect: " << SDLNet_GetError());
		return false;
	}

	m_sSocket = SDLNet_TCP_Open(&addr);
	if (m_sSocket == NULL)
	{
		DEBUG_PRINTL("Error TCP-Open: " << SDLNet_GetError());
		m_sSocket = NULL;
		return false;
	}

	DEBUG_PRINTL("connected to server " << szServer << " on port " << nPort);

	// Thread starten
	Start();

	/*if (bIsDedicated == true)	//wird gleich behandelt
	{*/
		DedicatedLogin();
	/*}
	else
	{
		LANLogin();
	}*/

	return true;
}

char* CClient::GetMapFile()
{
	return m_szCurrentMap;
}

int CClient::Run()
{
	while (!m_bTerminate)
	{
		DEBUG_PRINTL("ClientThread: Updating...");
		if (!Update())
		{
			Terminate();
		}
		SDL_Delay(m_nSendInterval);
	}
	return 0;
}

void CClient::Die()
{
	SRequestGamePlayerDie sRequest = { { 0, GAME_PLAYER_DIE } };
	SDLNet_Write16(sizeof(SRequestGamePlayerDie), &sRequest.sHeader.nMessageLength);

	AddMessage(new CMessage(sizeof(SRequestGamePlayerDie), (u8*) &sRequest));
}

bool CClient::Update()
{
	if (m_nAuthToken != 0)
	{
		// Serverzeit synchronisieren
		GenSyncMessage();
	}

	if (m_bInGame == true && m_sGameInfo.eState == SERVERGAME_RUNNING)
	{
		GenProgress();
	}

	int nSendResult = SendMsgQueue();
	if (nSendResult == -1)
	{
		return false;
	}
	else if (nSendResult == 1)
	{
		DEBUG_PRINTL("Checking for new messages");
		return RcvMsgs();
	}
	else
	{
		return true;
	}
}

int CClient::SendMsgQueue()
{
	if (m_qMsgs.size() < 1)
		return 0;

	SDL_LockMutex(m_pMutex);
	DEBUG_PRINTL("Committing " << m_qMsgs.size() << " Messages");

	SMessageHeader header;
	SDLNet_Write32(m_nAuthToken, &header.nAuthID);
	header.nNumMessages = m_qMsgs.size();
	SDLNet_Write32(header.nNumMessages, &header.nNumMessages);

	if(CMessage::SecureSend(m_sSocket, (u8*) &header, sizeof(SMessageHeader)) == false)
	{
		cerr << "->Error Sending Header: " << SDLNet_GetError() << endl;
		SDL_UnlockMutex(m_pMutex);
		return -1;	//das naechste mal wieder versuchen!
	}

	int i = 1;
	CMessage* pMsg;

	while (!m_qMsgs.empty())
	{
		pMsg = m_qMsgs.front();
		u8* pSendBuffer = pMsg->Data();
		DEBUG_PRINTL("->Sending Message #" << i << " - Type: " << (u16) ((SRequestHeader*) pSendBuffer)->eMessageType);

		if (!pMsg->SecureSend(m_sSocket, pSendBuffer, pMsg->GetSize()))
		{
			cerr << "->Error Sending Message!" << endl;
			SDL_UnlockMutex(m_pMutex);
			return -1;	//das naechste mal wieder versuchen!
		}

		FREEMEM(pSendBuffer);	//Wichtig! Den Puffer brauchen wir nicht mehr
		m_qMsgs.pop();
		i++;
	}

	SDL_UnlockMutex(m_pMutex);
	return 1;
}

bool CClient::RcvMsgs()
{
	DEBUG_PRINTL("Checking for new messages");
	SMessageHeader header;

	if (CMessage::SecureRecv(m_sSocket, (u8*) &header, sizeof(SMessageHeader)) == false)
	{
		cerr << "->Error Receiving Message!" << endl;
		return false;
	}

	header.nNumMessages = SDLNet_Read32(&header.nNumMessages);
	DEBUG_PRINTL("->" << header.nNumMessages << " waiting in queue");

	SResponseHeader* pRespHeader;
	CMessage* pRespMsg;
	u8* pRespBody;
	bool (CClient::*pCallback)(u8*);	//Callback-Methoden fuer einzelne Message-Typen. Parameter ist der pRespBody

	for (unsigned int i = 0; i < header.nNumMessages; i++)
	{
		DEBUG_PRINTL("->Receiving Message #" << i);

		//Header + Daten holen
		pRespMsg = new CMessage();

		if (!pRespMsg->RecvResponse(m_sSocket))
		{
			cerr << "->Error receiving response message!" << endl;
			return false;
		}

		pRespBody = pRespMsg->Data();
		pRespHeader = (SResponseHeader*) pRespBody;

		DEBUG_PRINTL("->Received Message Type " << (int) pRespHeader->eMessageType);
		DEBUG_PRINTL("->With " << pRespHeader->nMessageLength << " Bytes");
		DEBUG_PRINTL("->Return Code " << (int) pRespHeader->eReturnCode);

		//Je nach Message-Type eine Methode aufrufen
		switch (pRespHeader->eMessageType)
		{
			default:
				cerr << "->Error: Unknown message type" << endl;
				pCallback = NULL;
			break;

			case CLIENT_LOGIN:
				pCallback = &CClient::HandleClientLogin;
			break;

			case CLIENT_LOGOUT:
				pCallback = &CClient::HandleClientLogout;
			break;

			case CLIENT_SYNC:
				pCallback = &CClient::HandleClientSync;
			break;

			case SERVER_GET_GAMES:
				pCallback = &CClient::HandleServerGetGames;
			break;

			case SERVER_CREATE_GAME:
				pCallback = &CClient::HandleServerCreateGame;
			break;

			case GAME_MAP_CONFIG:
			{
				pCallback = NULL;	//fuer die kleine Nachricht brauchen wir keinen extra Handler
				SResponseGameMapConfig* pGameMap = (SResponseGameMapConfig*) pRespBody;
				if (pGameMap->sHeader.eReturnCode == CODE_OK)
				{
					DEBUG_PRINTL("->Map config succesfully");
				}
				else
				{
					cerr << "->Failed to configure maps" << endl;
				}
			}
			break;

			case GAME_JOIN:
				pCallback = &CClient::HandleGameJoin;
			break;

			case SERVER_GET_PLAYER:
			case GAME_GET_PLAYER:
				pCallback = &CClient::HandleGetPlayer;
			break;

			case GAME_GET_PLAYERINFO:
				pCallback = &CClient::HandleGamePlayerInfo;
			break;

			case GAME_START:
				pCallback = &CClient::HandleGameStart;
			break;

			case GAME_GET_MAP:
				pCallback = &CClient::HandleGameGetMap;
			break;

			case GAME_NEXT_LEVEL:
				pCallback = &CClient::HandleGameNextLevel;
			break;

			case GAME_PROGRESS:
				pCallback = &CClient::HandleGameProgress;
			break;
		}

		if (pCallback != NULL)
		{
			(*this.*pCallback)(pRespBody);
		}

		FREEMEM(pRespMsg);
	}

	/*if (header.nNumMessages > 0)
		Unblock();*/

	return true;
}

bool CClient::HandleClientLogin(u8* pRespBody)
{
	DEBUG_PRINTL("->Client login response received");

	SResponseClientLogin* pRespLogin = (SResponseClientLogin*) pRespBody;
	m_nAuthToken = SDLNet_Read32(&pRespLogin->nAuthID);
	m_nLocPlayerID = SDLNet_Read16(&pRespLogin->nID);
	DEBUG_PRINTL("->Login response authID: " << m_nAuthToken << " and Player-ID " << m_nLocPlayerID);

	return true;
}

bool CClient::HandleClientLogout(u8* pRespBody)
{
	DEBUG_PRINTL("->Client logout response received");

	m_bInGame = false;
	return true;
}

bool CClient::HandleClientSync(u8* pRespBody)
{
	DEBUG_PRINTL("->Client sync message received");
	SResponseClientSync* pResponse = (SResponseClientSync*) pRespBody;

	m_nLastServerTimeRecv = time(NULL);
	m_nServerTime = SDLNet_Read32(&pResponse->nServerTime);

	return true;
}

bool CClient::HandleServerGetGames(u8* pRespBody)
{
	DEBUG_PRINTL("->Receiving all games on the server...");

	SResponseServerGetGames* pServerGames = (SResponseServerGetGames*) pRespBody;
	pServerGames->nNumGames = SDLNet_Read16(&pServerGames->nNumGames);

	SGameInfo* pGameInfos = (SGameInfo*) (pRespBody + sizeof(SResponseServerGetGames));

	DEBUG_PRINTL("->Will receive " << pServerGames->nNumGames << " records...");
	vector<SGameInfo>::iterator iter;
	LockMutex();
	for (u16 i = 0; i < pServerGames->nNumGames; i++)
	{
		pGameInfos[i].nID = SDLNet_Read16(&pGameInfos[i].nID);
		iter = m_vAvailableGames.begin();
		while (iter != m_vAvailableGames.end() && iter->nID != pGameInfos[i].nID)
		{
			iter++;
		}

		if (iter == m_vAvailableGames.end())
		{
			// Neues Spiel -> Zum vektor hinzufuegen
			m_vAvailableGames.push_back(pGameInfos[i]);
		}
		else
		{
			// Vorhandenes Spiel -> akutalisieren
			(*iter) = pGameInfos[i];
		}

		DEBUG_PRINTL("Game: " << pGameInfos[i].szName << " with id " << pGameInfos[i].nID);
		DEBUG_PRINT(" - " << (u16) pGameInfos[i].nNumPlayers << "/" << (u16) pGameInfos[i].nMaxPlayers << " Player");
		if (pGameInfos[i].eState == SERVERGAME_RUNNING)
		{
			DEBUG_PRINT(" - running");
		}
		DEBUG_PRINTL(" ");
	}
	UnlockMutex();

	m_nLastRefresh = SDL_GetTicks();
	return true;
}

bool CClient::HandleGetPlayer(u8* pRespBody)
{
	DEBUG_PRINTL("->Receiving all players in the game...");

	SResponseGameGetPlayer* pResponse = (SResponseGameGetPlayer*) pRespBody;
	pResponse->nNumPlayer = SDLNet_Read16(&pResponse->nNumPlayer);

	SPlayerInfo* pInfos = (SPlayerInfo*) (pRespBody + sizeof(SResponseGameGetPlayer));

	DEBUG_PRINTL("->Will receive " << pResponse->nNumPlayer << " records...");

	vector<SPlayer>::iterator iter;
	LockMutex();

	SPlayer sPlayer;
	for (u16 i = 0; i < pResponse->nNumPlayer; i++)
	{
		pInfos[i].nID = SDLNet_Read16(&pInfos[i].nID);
		sPlayer.sInfo = pInfos[i];

		iter = m_vPlayers.begin();
		while (iter != m_vPlayers.end() && iter->sInfo.nID != sPlayer.sInfo.nID)
		{
			//DEBUG_PRINTL(iter->sInfo.nID << ", " << nID);
			iter++;
		}

		if (iter == m_vPlayers.end())
		{
			// Neuer Spieler -> Zum vektor hinzufuegen
			m_vPlayers.push_back(sPlayer);
		}
		else
		{
			// Vorhandener Spieler -> akutalisieren
			iter->sInfo = sPlayer.sInfo;
		}

		DEBUG_PRINTL("Player: " << pInfos[i].szName << " with id " << &pInfos[i].nID);
	}

	if (pResponse->nNumPlayer < m_vPlayers.size())
	{
		// Alte Spielerdaten aus der map entfernen
		bool isOldEntry = false;
		for (iter = m_vPlayers.begin(); iter != m_vPlayers.end();)
		{
			isOldEntry = true;
			for (int j = 0; j < pResponse->nNumPlayer; j++)
			{
				if (pInfos[j].nID == iter->sInfo.nID)
				{
					isOldEntry = false;
					break;
				}
			}

			if (isOldEntry == true)
			{
				iter = m_vPlayers.erase(iter);
			}
			else
			{
				iter++;
			}
		}
	}

	UnlockMutex();

	m_nLastRefresh = SDL_GetTicks();
	return true;
}

bool CClient::HandleGamePlayerInfo(u8* pRespBody)
{
	DEBUG_PRINTL("->Receiving all players ingame info...");

	SResponseGameGetPlayerInfo* pResponse = (SResponseGameGetPlayerInfo*) pRespBody;
	pResponse->nNumPlayer = SDLNet_Read16(&pResponse->nNumPlayer);

	u16* pOrder = (u16*) (pRespBody + sizeof(SResponseGameGetPlayerInfo));
	SGamePlayerInfo* pInfos = (SGamePlayerInfo*) (pOrder + pResponse->nNumPlayer);

	DEBUG_PRINTL("->Will receive " << pResponse->nNumPlayer << " records...");

	vector<SPlayer>::iterator iter;
	u16 nID;
	LockMutex();
	for (u16 i = 0; i < pResponse->nNumPlayer; i++)
	{
		iter = m_vPlayers.begin();
		nID = SDLNet_Read16(&pOrder[i]);
		// Suche den passenden Spieler zur ID, die in pOrder angegeben ist
		while (iter != m_vPlayers.end() && iter->sInfo.nID != nID)
		{
			//DEBUG_PRINTL(iter->sInfo.nID << ", " << nID);
			iter++;
		}

		if (iter == m_vPlayers.end())
		{
			continue;
		}

		iter->sIngameInfo = pInfos[i];
		DEBUG_PRINTL("Player: " << iter->sInfo.szName << " with id " << iter->sInfo.nID);
	}
	UnlockMutex();

	m_nLastRefresh = SDL_GetTicks();
	return true;
}


bool CClient::HandleGameProgress(u8* pRespBody)
{
	DEBUG_PRINTL("->Receiving all players progress info...");

	SResponseGameProgress* pResponse = (SResponseGameProgress*) pRespBody;
	pResponse->nNumPlayer = SDLNet_Read16(&pResponse->nNumPlayer);

	u16* pOrder = (u16*) (pRespBody + sizeof(SResponseGameProgress));
	SGamePlayerStats* pInfos = (SGamePlayerStats*) (pOrder + pResponse->nNumPlayer);

	DEBUG_PRINTL("->Will receive " << pResponse->nNumPlayer << " records...");

	SPlayerObj sPlayer;
	map<int, SPlayerObj>::iterator iter;
	LockMutex();
	for (u16 i = 0; i < pResponse->nNumPlayer; i++)
	{
		/// Lokalen Spieler nicht zu map hinzufuegen
		if (GetLocalPlayerID() == SDLNet_Read16(&pOrder[i]))
		{
			DEBUG_PRINTL("->Skippin local player record...");
			continue;
		}

		pOrder[i] = SDLNet_Read16(&pOrder[i]);

		memset(&sPlayer, 0, sizeof(SPlayerObj));
		sPlayer.nID = pOrder[i];
		sPlayer.fAngleX = pInfos[i].fAngleX;
		sPlayer.fAngleY = pInfos[i].fAngleY;
		sPlayer.fAngleZ = pInfos[i].fAngleZ;

		if ((iter = m_mPlayerObjs.find(sPlayer.nID)) == m_mPlayerObjs.end())	//Wenn der Spieler noch nicht existiert die folgenden Daten setzen...
		{
			sPlayer.fCrashAnimTimer = CRASH_ANIM_SECONDS;
			sPlayer.fDeathAnimTimer = DEATH_ANIM_SECONDS;
			sPlayer.bDead = false;
		}
		else //... sonst die Daten in Ruhe lassen, d.h. alte Werte uebernehmen
		{
			sPlayer.fCrashAnimTimer = iter->second.fCrashAnimTimer;
			sPlayer.fDeathAnimTimer = iter->second.fDeathAnimTimer;
			sPlayer.bDead = iter->second.bDead;
		}

		sPlayer.fDmg = pInfos[i].fDamage;
		sPlayer.fFuel = pInfos[i].fFuel;
		sPlayer.fPosX = pInfos[i].fPosX;
		sPlayer.fPosY = pInfos[i].fPosY;
		sPlayer.fPosZ = pInfos[i].fPosZ;
		sPlayer.fSpeedX = pInfos[i].fSpeedX;
		sPlayer.fSpeedZ = pInfos[i].fSpeedZ;
		sPlayer.PlayerColl.vTestPoint[0].x = pInfos[i].sPlayerColl.vTestPoint[0];
		sPlayer.PlayerColl.vTestPoint[0].y = pInfos[i].sPlayerColl.vTestPoint[1];
		sPlayer.PlayerColl.vTestPoint[1].x = pInfos[i].sPlayerColl.vTestPoint[2];
		sPlayer.PlayerColl.vTestPoint[1].y = pInfos[i].sPlayerColl.vTestPoint[3];
		sPlayer.PlayerColl.vTestPoint[2].x = pInfos[i].sPlayerColl.vTestPoint[4];
		sPlayer.PlayerColl.vTestPoint[2].y = pInfos[i].sPlayerColl.vTestPoint[5];
		sPlayer.PlayerColl.vTestPoint[3].x = pInfos[i].sPlayerColl.vTestPoint[6];
		sPlayer.PlayerColl.vTestPoint[3].y = pInfos[i].sPlayerColl.vTestPoint[7];

		sPlayer.PlayerColl.fPlayerHeight = PLAYER_COLL_HEIGHT;
		sPlayer.PlayerColl.fPlayerWidth = PLAYER_COLL_WIDTH;
		sPlayer.PlayerColl.fPlayerLength = PLAYER_COLL_LENGTH;

		//den richtigen Namen heraussuchen...
		for (unsigned int j = 0; j < m_vPlayers.size(); j++)
		{
			if (m_vPlayers[j].sInfo.nID == sPlayer.nID)
			{
				sPlayer.szName = m_vPlayers[j].sInfo.szName;
			}
		}

		m_mPlayerObjs[sPlayer.nID] = sPlayer;

		DEBUG_PRINTL("Player Stats from: " << sPlayer.nID);
	}

	if (pResponse->nNumPlayer < m_mPlayerObjs.size())
	{
		// Alte Spielerdaten aus der map entfernen
		bool isOldEntry = false;
		for (iter = m_mPlayerObjs.begin(); iter != m_mPlayerObjs.end();)
		{
			isOldEntry = true;
			for (int j = 0; j < pResponse->nNumPlayer; j++)
			{
				if (pOrder[j] == iter->second.nID)
				{
					isOldEntry = false;
					break;
				}
			}

			if (isOldEntry == true)
			{
				m_mPlayerObjs.erase(iter++);
			}
			else
			{
				++iter;
			}
		}
	}

	UnlockMutex();

	m_nLastRefresh = SDL_GetTicks();
	return true;
}

bool CClient::HandleGameJoin(u8* pRespBody)
{
	DEBUG_PRINTL("->Game join response received");

	SResponseGameJoin* pGameJoin = (SResponseGameJoin*) pRespBody;
	DEBUG_PRINTL("->Return Code: " << (int) pGameJoin->sHeader.eReturnCode);

	if (pGameJoin->sHeader.eReturnCode != CODE_OK)
	{
		m_nCurrGameID = -1;
		m_bInGame = false;
		Unblock();
		return false;
	}
	else
	{
		m_bInGame = true;
		Unblock();
		return true;
	}
}

bool CClient::HandleServerCreateGame(u8* pRespBody)
{
	SResponseServerCreateGame* pCreateGame = (SResponseServerCreateGame*) pRespBody;

	if (pCreateGame->sHeader.eReturnCode == CODE_OK)
	{
		m_nCurrGameID = SDLNet_Read16(&pCreateGame->nGameID);
		m_bInGame = true;
		DEBUG_PRINTL("->Create game response GameID: " << (int) m_nCurrGameID);
		return true;
	}
	else
	{
		m_nCurrGameID = -1;
		DEBUG_PRINTL("->Create game response failed. Return Code: " << (int) pCreateGame->sHeader.eReturnCode);
		return false;
	}
}

bool CClient::HandleGameStart(u8* pRespBody)
{
	DEBUG_PRINTL("->Game start response received");

	SResponseGameStart* pResponse = (SResponseGameStart*) pRespBody;
	DEBUG_PRINTL("->Return Code: " << (int) pResponse->sHeader.eReturnCode);

	if (pResponse->sHeader.eReturnCode != CODE_OK)
		return false;
	else
		return true;
}


bool CClient::HandleGameGetMap(u8* pRespBody)
{
	DEBUG_PRINTL("->Game get map response received");

	SResponseGameGetMap* pGetMap = (SResponseGameGetMap*) pRespBody;

	if (pGetMap->sHeader.eReturnCode == CODE_OK)
	{
		pGetMap->nLength = SDLNet_Read16(&pGetMap->nLength);
		int nFileSize = 0;

		m_szCurrentMap = tmpnam(NULL);

		DEBUG_PRINTL("->Creating tempfile " << m_szCurrentMap);
		FILE* fd = fopen(m_szCurrentMap, "w");
		if (fd != NULL)
		{
			u8* pData = pRespBody + sizeof(SResponseGameGetMap);
			nFileSize += fwrite(pData + nFileSize, 1, pGetMap->nLength, fd);
			fclose(fd);
			cout << "->Map received succesfully" << endl;

			return true;
		}
		else
		{
			m_szCurrentMap = NULL;
			cerr << "->Error creating tempfile!" << endl;
			return false;
		}
	}

	return false;
}

bool CClient::HandleGameNextLevel(u8* pRespBody)
{
	SResponseGameNextLevel* pResponse = (SResponseGameNextLevel*) pRespBody;

	if (pResponse->sHeader.eReturnCode == CODE_OK)
	{
        map<int, SPlayerObj>::iterator iter;
        for (iter = m_mPlayerObjs.begin(); iter != m_mPlayerObjs.end(); iter++)
        {
            iter->second.bDead = false;
        }

		m_nPlayerStartPos = pResponse->nPlayerStartPos;
		m_sGameInfo = pResponse->sGameInfo;
		m_nNextLevelStartTime = SDLNet_Read32(&pResponse->nStartTime);
		DEBUG_PRINTL("GAME now running: " << (u16) m_sGameInfo.eState);
		return true;
	}
	else
	{
		return false;
	}
}

int CClient::GetLocalPlayerID()
{
	return m_nLocPlayerID;
}

int CClient::GetCurrGameID()
{
	return m_nCurrGameID;
}

vector<SGameInfo>& CClient::GetGames()
{
	if (m_nLastRefresh + m_nRefreshInterval < SDL_GetTicks())
	{
		GenServerGetGames();
	}

	return m_vAvailableGames;
}

vector<SPlayer>& CClient::GetPlayer()
{
	if (m_nLastRefresh + m_nRefreshInterval < SDL_GetTicks())
	{
		GenGetPlayer();
	}

	return m_vPlayers;
}

/*map<int, SPlayerInfo*>& CClient::GetPlayerStats()
{
	if (m_nLastRefresh + m_nRefreshInterval < SDL_GetTicks())
	{
		GenGetPlayer();
	}

	LockMutex();
	m_mPlayerStats.clear();
	for (int i = 0; i < m_vPlayers.size(); i++)
	{
		m_mPlayerStats[m_vPlayers[i].sIngameInfo.nRoundsWon] = &m_vPlayers[i].sInfo;
	}
	UnlockMutex();

	return m_mPlayerStats;
}*/

bool CClient::SortPlayerStats(SPlayer p1, SPlayer p2)
{
	return p1.sIngameInfo.nRoundsWon > p2.sIngameInfo.nRoundsWon;
}

map<int, SPlayerObj>* CClient::GetPlayerObj()
{
	return &m_mPlayerObjs;
}

void CClient::SetLocalPlayerObj(SPlayerObj* pPlayer)
{
	m_pLocalPlayer = pPlayer;
}

SGameInfo* CClient::GetRunningGame()
{
	// Nimm Client ueberhaupt an einem Spiel teil?
	if (m_bInGame == false)
	{
		return NULL;
	}

	/*if (m_pGameInfo == NULL || m_nLastRefresh + m_nRefreshInterval < SDL_GetTicks())
	{

	}*/

	return &m_sGameInfo;
}

void CClient::GenClientLogin()
{
	const char* szPlayerName = m_pGame->GetSavedPlayerData()->sPlayerName.c_str();

	// Login Nachricht erstellen
	CMessage* pMessage = new CMessage(sizeof(SRequestClientLogin));
	SRequestClientLogin* pRequest = (SRequestClientLogin*) pMessage->Data();

	SDLNet_Write16(sizeof(SRequestClientLogin), &pRequest->sHeader.nMessageLength);
	pRequest->sHeader.eMessageType = CLIENT_LOGIN;

	memcpy(pRequest->szName, szPlayerName, PLAYER_NAME_LENGTH);

	AddMessage(pMessage);
}

void CClient::GenServerGetGames()
{
	SRequestServerGetGames sRequest = { {0, SERVER_GET_GAMES }, true, false, 0, 0 };
	SDLNet_Write16(sizeof(SRequestServerGetGames), &sRequest.sHeader.nMessageLength);

	CMessage* pMessage = new CMessage(sizeof(SRequestServerGetGames), (u8*) &sRequest);

	AddMessage(pMessage);

	m_nLastRefresh = SDL_GetTicks();	//Wichtig! 1 Request pro Refresh-Zeitscheibe reicht schliesslich...
}

void CClient::GoalReached()
{
	SRequestGameNextLevel sRequest = { {0, GAME_NEXT_LEVEL } };
	SDLNet_Write16(sizeof(SRequestGameNextLevel), &sRequest.sHeader.nMessageLength);

	CMessage* pMessage = new CMessage(sizeof(SRequestGameNextLevel), (u8*) &sRequest);
	AddMessage(pMessage);
}

void CClient::GenSyncMessage()
{
	SRequestClientSync sRequest = { { 0, CLIENT_SYNC } };
	SDLNet_Write16(sizeof(SRequestClientSync), &sRequest.sHeader.nMessageLength);

	CMessage* pMessage = new CMessage(sizeof(SRequestClientSync), (u8*) &sRequest);
	AddMessage(pMessage);
}

void CClient::GenProgress()
{
	if (m_pLocalPlayer == NULL || m_nLastSend + m_nSendInterval > SDL_GetTicks())
	{
		return;
	}

	SRequestGamePushProgress sRequest;
	sRequest.sHeader.eMessageType = GAME_PUSH_PROGRESS;
	SDLNet_Write16(sizeof(SRequestGamePushProgress), &sRequest.sHeader.nMessageLength);

	sRequest.sStats.fAngleX = m_pLocalPlayer->fAngleX;
	sRequest.sStats.fAngleY = m_pLocalPlayer->fAngleY;
	sRequest.sStats.fAngleZ = m_pLocalPlayer->fAngleZ;
	sRequest.sStats.fDamage = m_pLocalPlayer->fDmg;
	sRequest.sStats.fFuel = m_pLocalPlayer->fFuel;
	sRequest.sStats.fPosX = m_pLocalPlayer->fPosX;
	sRequest.sStats.fPosY = m_pLocalPlayer->fPosY;
	sRequest.sStats.fPosZ = m_pLocalPlayer->fPosZ;
	sRequest.sStats.fSpeedX = m_pLocalPlayer->fSpeedX;
	sRequest.sStats.fSpeedZ = m_pLocalPlayer->fSpeedZ;
	sRequest.sStats.sPlayerColl.vTestPoint[0] = m_pLocalPlayer->PlayerColl.vTestPoint[0].x;
	sRequest.sStats.sPlayerColl.vTestPoint[1] = m_pLocalPlayer->PlayerColl.vTestPoint[0].y;
	sRequest.sStats.sPlayerColl.vTestPoint[2] = m_pLocalPlayer->PlayerColl.vTestPoint[1].x;
	sRequest.sStats.sPlayerColl.vTestPoint[3] = m_pLocalPlayer->PlayerColl.vTestPoint[1].y;
	sRequest.sStats.sPlayerColl.vTestPoint[4] = m_pLocalPlayer->PlayerColl.vTestPoint[2].x;
	sRequest.sStats.sPlayerColl.vTestPoint[5] = m_pLocalPlayer->PlayerColl.vTestPoint[2].y;
	sRequest.sStats.sPlayerColl.vTestPoint[6] = m_pLocalPlayer->PlayerColl.vTestPoint[3].x;
	sRequest.sStats.sPlayerColl.vTestPoint[7] = m_pLocalPlayer->PlayerColl.vTestPoint[3].y;

	DEBUG_PRINTL("Send own progress message");
	m_nLastSend = SDL_GetTicks();
	AddMessage(new CMessage(sizeof(SRequestGamePushProgress), (u8*) &sRequest));
}

bool CClient::CreateGame(const char* szName, u8 nMaxPlayers,
				bool bAllowLateJoin, bool bAllowVisitors,
				const char* szPassword, vector< u8 >& vSelectedMaps)
{
	//CreateGame Msg erstellen
	SRequestServerCreateGame* pCreateGame = new SRequestServerCreateGame;
	pCreateGame->sHeader.eMessageType = SERVER_CREATE_GAME;

	memcpy(pCreateGame->szGameName, szName, GAME_NAME_LENGTH);
	pCreateGame->nMaxPlayer = (u8) nMaxPlayers;
	pCreateGame->bAllowLateJoin = bAllowLateJoin;
	pCreateGame->bAllowVisitors = bAllowVisitors;
	memcpy(pCreateGame->szPassword, szPassword, GAME_PASSWD_LENGTH);

	SDLNet_Write16(sizeof(SRequestServerCreateGame), &pCreateGame->sHeader.nMessageLength);

	u8* pBuffer = (u8*) pCreateGame;	//Fragt mich nicht, warum DAS noetig ist...
	CMessage* pMsg = new CMessage(sizeof(SRequestServerCreateGame), pBuffer);
	FREEMEM(pCreateGame);	//CMessage kopiert den Puffer, daher ist das hier wichtig!

	AddMessage(pMsg);

	//Maps hinzufuegen
	for (unsigned int i = 0; i < vSelectedMaps.size(); i++)
	{
		AddMap(vSelectedMaps[i]);
	}

	return true;
}

void CClient::AddMap(u8 nMapID)
{
	SRequestGameMapConfig sRequest = { { 0, GAME_MAP_CONFIG }, REQUEST_ADD_MAP, 1, nMapID };
	SDLNet_Write16(sizeof(SRequestGameMapConfig), &sRequest.sHeader.nMessageLength);

	u8* pBuffer = (u8*) &sRequest;
	CMessage* pMsg = new CMessage(sizeof(SRequestGameMapConfig), pBuffer);

	AddMessage(pMsg);
}


bool CClient::JoinGame(u16 nID, const char* szPassword)
{
	GenGameJoin(nID, szPassword);
	m_nCurrGameID = nID;
	//SDL_Delay(500);
	//SendMsgQueue();
	WaitResponse();
	GenGetPlayer();

	return true;
}

bool CClient::LeaveGame()
{
	m_nCurrGameID = -1;

	SRequestGameLeave sRequest;
	sRequest.sHeader.eMessageType = GAME_LEAVE;

	SDLNet_Write16(sizeof(SRequestGameLeave), &sRequest.sHeader.nMessageLength);

	CMessage* pMessage = new CMessage(sizeof(SRequestGameLeave), (u8*) &sRequest);

	AddMessage(pMessage);

	return true;

}

bool CClient::StartGame()
{
	SRequestGameStart sRequest;
	sRequest.sHeader.eMessageType = GAME_START;

	SDLNet_Write16(sizeof(SRequestGameStart), &sRequest.sHeader.nMessageLength);

	CMessage* pMessage = new CMessage(sizeof(SRequestGameStart), (u8*) &sRequest);

	AddMessage(pMessage);

	return true;
}

bool CClient::LevelRunning()
{
	if (InGame() == true)
	{
		if (m_nServerTime + (time(NULL) - m_nLastServerTimeRecv) > m_nNextLevelStartTime)
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	return false;
}

int CClient::SecondsToStart()
{
	return m_nNextLevelStartTime - (m_nServerTime + (time(NULL) - m_nLastServerTimeRecv));
}

void CClient::GenGameJoin(u16 nID, const char* szPassword)
{
	SRequestGameJoin sRequest;
	sRequest.sHeader.eMessageType = GAME_JOIN;

	SDLNet_Write16(sizeof(SRequestGameJoin), &sRequest.sHeader.nMessageLength);
	SDLNet_Write16(nID, &sRequest.nGameID);

	if (szPassword)
		memcpy(&sRequest.szPassword, szPassword, strlen(szPassword) + 1);

	CMessage* pMessage = new CMessage(sizeof(SRequestGameJoin), (u8*) &sRequest);

	AddMessage(pMessage);
}


void CClient::GenGetPlayer()
{
	DEBUG_PRINTL("Generating *_GET_PLAYER request...");

	if (m_bInGame == false)
	{
		// Spieler auf dem Server abrufen die noch keinem Spiel beigetreten sind
		// Keine Spieler-InGame-Daten abrufen
		SRequestServerGetPlayer sRequest = { { 0, SERVER_GET_PLAYER } };
		SDLNet_Write16(sizeof(SRequestServerGetPlayer), &sRequest.sHeader.nMessageLength);

		AddMessage(new CMessage(sizeof(SRequestServerGetPlayer), (u8*) &sRequest));
	}
	else
	{
		// Spieler-Infos ueber Mitspieler und InGame-Infos
		SRequestServerGetPlayer sRequest = { { 0, GAME_GET_PLAYER } };
		SDLNet_Write16(sizeof(SRequestServerGetPlayer), &sRequest.sHeader.nMessageLength);

		AddMessage(new CMessage(sizeof(SRequestServerGetPlayer), (u8*) &sRequest));

		SRequestGameGetPlayerInfo sIngameRequest = { { 0, GAME_GET_PLAYERINFO } };
		SDLNet_Write16(sizeof(SRequestGameGetPlayerInfo), &sIngameRequest.sHeader.nMessageLength);

		AddMessage(new CMessage(sizeof(SRequestGameGetPlayerInfo), (u8*) &sIngameRequest));
	}

	m_nLastRefresh = SDL_GetTicks();	//Wichtig! 1 Request pro Refresh-Zeitscheibe reicht schliesslich...
}

void CClient::GenServerGetMaps()
{
	SRequestServerGetMaps sRequest = { {0, SERVER_GET_MAPS } };
	SDLNet_Write16(sizeof(SRequestServerGetMaps), &sRequest.sHeader.nMessageLength);
	CMessage* pMessage = new CMessage(sizeof(SRequestServerGetMaps), (u8*) &sRequest);

	AddMessage(pMessage);
}

bool CClient::DedicatedLogin()
{
	// Erst anmelden
	GenClientLogin();

	// Dann Informationen ueber Spieler und Spiele auf dem Server abrufen
	GenServerGetGames();
	//WaitResponse();
	//GenGetPlayer();	//brauchen wir momentan nicht

	return true;
}

/*bool CClient::LANLogin()
{
	// Erst anmelden
	GenClientLogin();

	// Dem Default LAN-Spiel beitreten
	JoinGame(LAN_GAME_ID, NULL);
	WaitResponse();

	// Wen dem Spiel beigereten wurde können Spieler-Infos abgerufen werden
	if (this->m_bInGame == true)
	{
		GenGetPlayer();
		return true;
	}
	else
	{
		return false;
	}
}*/

void CClient::AddMessage(CMessage* pMessage)
{
	SDL_LockMutex(m_pMutex);
	m_qMsgs.push(pMessage);
	SDL_UnlockMutex(m_pMutex);
}

void CClient::WaitResponse()
{
	Block();
	while (m_bBlock == true)
	{
		DEBUG_PRINTL("WaitResponse(): Blocking...");
		SDL_Delay(m_nSendInterval);
	}
}
