#ifndef CLIENT_HPP_
#define CLIENT_HPP_

#include "../tools/BaseThread.hpp"
#include "../net/SocketFramework.hpp"
#include "../net/NetworkTypes.hpp"
#include "../net/Message.hpp"
#include "../game/Game.hpp"
#include "../game/PlayerCtrlMP.hpp"

#include <queue>
#include <vector>
#include <string>

#define CLIENT_REFRESH_INTERVAL 100
#define CLIENT_SEND_INTERVAL 100
#define CLIENT_MSG_QUEUE_LEN 10

using namespace std;

class CGame;
class CPlayerCtrlMP;
class CClientThread;

struct SPlayer
{
	SPlayerInfo sInfo;
	SGamePlayerInfo sIngameInfo;
};

/// Kommunikation mit einem Server fuer ein Netzwerkspiel
/// Empfaengt regelmaessig Daten ueber Spielverlauf und aktualisiert
/// PlayerCtrlMP-Objekt

class CClient : public BaseThread
{
	public:
		/// Konstruktor
		/// @param	pLocalPlayer	Lokaler player
		/// @param	pGame	CGame-Instanz
		/// @param	nRefreshInterval	Intervall in dem Informationen aktualisiert werden
		/// @param	nSendInterval	Interval in dem der aktualle (in ms)
		/// 									Spielfortschritt abgerufen wird und der
		/// 									eigene Fortschritt gesendet wird (in ms)
		/// @param	nMessageQueueLength	Anzahl von Chat-Nachrichten die gespeichert werden
		CClient(CGame* pGame, unsigned int nRefreshInterval,
				unsigned int nSendInterval, unsigned int nMessageQueueLength);
		
		~CClient();

		/// Zeiger auf PlayerCtrlMP Objekt setzen
		void SetGameController(CPlayerCtrlMP* pGameController) { m_pLocalGameController = pGameController; }

		/// ID des lokalen Spielers zurueckgeben
		int GetLocalPlayerID();

		///ID des aktuellen Spiels zurueckgeben
		int GetCurrGameID();

		///Startposition des Spielers auf der aktuellen Map zurueck geben

		int GetPlayerStartPos() { return m_nPlayerStartPos; }
		
		/// Gib den Dateinamen der vom Server empfangenen Map zurück

		char* GetMapFile();

		/// Verbindung zu einem Server herstellen
		/// @param	szServer		Name des Servers
		/// @param	nPort			Port auf dem Server
		/// @return	true wenn eine Verbindung hergestellt werden konnte sonst
		/// 			false

		bool connect(const char* szServer, int nPort);
		
		/// Gibt Informationen ueber alle auf dem Server verfuegbaren Spiele
		/// zurueck.
		/// @return	SGameInfo-Vektor

		vector< SGameInfo >& GetGames();
		
		/// Gibt Information ueber das Spiel zurueck an dem der Spieler
		/// beteiligt ist
		/// @return	Spielinformationen

		SGameInfo* GetRunningGame();
		
		/// Gibt Informationen ueber im Spiel oder auf dem Server registrierte
		/// Spieler zurueck. Sind noch keine Infos vom Server empfangen worden,
		/// blockiert die Funktion und ruft diese ab.
		/// @param	pnNumPlayer	Zeiger auf desses Ziel die Anzahl gelesener
		/// 						Spieler-Infos gespeichert wird
		/// @return	Einen Zeiger auf ein SPlayerInfo-Array

		vector<SPlayer>& GetPlayer();

		/// Generierte eine Msg, die neue Spielerstats anfordert und gib akt. Spielerstats
		/// zurueck als sortierte Map mit <Anzahl gew. Spiele, Spielerinfo>

		map< int, SPlayerInfo* >& GetPlayerStats();
		
		/// Gibt Informationen ueber im Spiel oder auf dem Server registrierte
		/// Spieler zurueck.
		/// @return	Einen Pointer auf eine Map mit SPlayerObj-Strukturen

		map<int, SPlayerObj>* GetPlayerObj();

		/// Setzt die Referenz zum lokalen player
		/// @param	pPlayer

		void SetLocalPlayerObj(SPlayerObj* pPlayer);
		
		/// Gibt Informationen zu verfuegbaren Maps zurueck die auf dem Server
		/// sind und abgerufen werden koennen.
		/// Sind noch keine Infos vom Server empfangen worden,
		/// blockiert die Funktion und ruft diese ab.
		/// @param	pnNumMaps	Zeiger auf desses Ziel die Anzahl gelesener
		/// 						Level-Infos gespeichert wird
		/// @return	Einen Zeiger auf ein SMapInfo-Array

		SMapInfo* GetMaps(u16* pnNumMaps);
		
		/// Laedt Level-Daten fuer das aktuelle Level vom Server herunter und
		/// gibt einen Zeiger auf diese Daten zurueck
		/// @param	pnSize	Zeiger auf dessen Ziel die Groesse der Karte
		/// 					gespeichert wird
		/// @return Zeiger auf Kartendaten

		u8*	GetMap();

		/// Fuegt eine Map zur "Playlist" des Servers hinzu

		void AddMap(u8 nMapID);
		
		/// Sendet eine Chat-Nachricht
		/// @param	szMessage	zu sendende Chat-Nachricht

		void Chat(char* szMessage);
		
		/// Gibt einen Vector mit eingegangenen Chat-Nachrichten zurueck
		/// @return	Nachrichten-Vektor

		vector<string>& GetChatMessages();
		
		/// Erstellt ein neues Spiel
		/// @param	szName	Name des Spiels
		/// @param	nMaxPlayers	Maximale Anzahl Spieler
		/// @param	bAllowLateJoin	Erlaube das Beitreten von Spielern nachdem
		/// 							das Spiel gestartet wurde?
		/// @param	bAllowVisitors	Zuschauer erlauben?
		/// @param	szPassword	Das Passwort (NULL für kein Passwort)
		/// @return	true wenn Spiel erstellt wurde sonst false

		bool CreateGame(const char* szName, u8 nMaxPlayers, 
				bool bAllowLateJoin, bool bAllowVisitors,
				const char* szPassword, vector< u8 >& vSelectedMaps);
		
		/// Tritt einem Spiel bei
		/// @param	nID	ID des Spiels
		/// @param	szPassword	Passwort (NULL fuer kein Passwort)
		/// @return	true wenn beigetreten werden konnte sonst false

		bool JoinGame(u16 nID, const char* szPassword);

		/// Setzt Spieler auf PLAYER_READY und startet Spiel sobald alle bereit sind
		/// @return	true wenn gestartet werden konnte

		bool StartGame();
		
		/// Tritt einem Spiel als Zuschauer bei
		/// @param	nID	ID des Spiels
		/// @param	szPassword	Passwort (NULL fuer kein Passwort)
		/// @return	true wenn beigetreten werden konnte sonst false

		bool VisitGame(u16 nID, char* szPassword);
		
		/// Verlaesst das Spiel an dem der Client gerade beteiligt ist

		bool LeaveGame();
		
		/// Pausiert das spiel an dem der Client gerade beteiligt ist

		void PauseGame();

		///Spieler hat das Ziel erreicht -> GAME_NEXT_LEVEL Nachricht schicken!
		void GoalReached();

		void LockMutex() { SDL_LockMutex(m_pMutex); }
		void UnlockMutex() { SDL_UnlockMutex(m_pMutex); }
		
		/// Gibt true zurueck wenn Client gerade an einem Spiel beteiligt ist
		/// @return true|false

		bool InGame() { return m_bInGame; }
		
		/// Gibt true zurück wenn ein level geladen ist und der counter zum
		/// levelstart abgelaufen ist

		bool LevelRunning();
		
		/// Gib die Anzahl sekunden zurück bis zum start

		int SecondsToStart();
		
		/// Prueft ob die Verbindung noch besteht und evtl. Fehler aufgetreten sind
		/// @return	true wenn die Verbindung noch besteht, false wenn ein
		/// 			kritischer Fehler zum Verbindungsabbau gefuehrt hat

		bool IsActive();
		
		/// Liefert den zuletzt aufgetretenen Fehler zurueck
		/// @return	Fehler-String des letzten Fehlers oder NULL wenn kein Fehler
		/// 			aufgetreten ist

		char* LastError();
		
		/// Endlosscheife in der regelmaessig Daten gesendet werden

		int Run();

		/// Daten empfangen und senden. Wird vom ClientThread in regelmaessigen Abstaenden
		/// aufgerufen
		bool Update();

		/// Aufgerufen wenn Spieler tot ist
		void Die();
		
		/// Sort-Callback fuer Sortierung der Spielerstats nach gew. Runden
		static bool SortPlayerStats(SPlayer p1, SPlayer p2);
		
	private:
		/// Socket
		TCPsocket m_sSocket;
		
		/// Verfuegbare Spiele
		CMessage* m_pRemoteGames;
		
		/// Registrierte Spieler auf dem Server / Im Spiel
		CMessage* m_pRemotePlayer;
		
		/// Verfuegbare Level-Karten
		CMessage* m_pRemoteMaps;

		/// Lokaler Spieler
		SPlayerObj* m_pLocalPlayer;
		
		/// Vektor von Spielerdaten
		map< int, SPlayerObj > m_mPlayerObjs;
		
		/// Aktuelle Zeit auf dem Server
		int m_nServerTime;
		int m_nLastServerTimeRecv;
		
		/// Flag zum signalisieren ob Client an Spiel teilnimmt
		bool m_bInGame;
		
		/// Informationen zum Spiel an dem der Client gerade teilnimmt
		SGameInfo m_sGameInfo;
		//CMessage* m_pGameInfo;
		
		/// Referenz zum Spiele-Controller
		CPlayerCtrlMP* m_pLocalGameController;
		
		/// Mutex
		SDL_mutex* m_pMutex;
		
		/// Queue zu sendender Nachrichten
		queue<CMessage*> m_qMsgs;

		/// Vektor mit verfuegbaren Spielen auf dem aktuellen Server
		vector< SGameInfo > m_vAvailableGames;

		/// Vektor mit den Spielern
		vector< SPlayer > m_vPlayers;

		/// Sortierte Map mit Spielern und deren Anzahl gewonnener Spiele
		//map< int, SPlayerInfo* > m_mPlayerStats;
		
		/// Ausfuehrung blockieren bis ein Resultat Daten gesendet wurden und
		/// eine Antwort erhalten wurde
		bool m_bBlock;
		
		/// AuthID fuer diesen Client
		int m_nAuthToken;

		/// ID fuer diesen Client
		int m_nLocPlayerID;

		/// ID fuer das aktuelle Spiel
		int m_nCurrGameID;

		/// zeit wann das neue Level gestartet werden soll
		int m_nNextLevelStartTime;
		
		/// Start-Position des Spielers auf der aktuellen Map
		int m_nPlayerStartPos;
		
		/// Wartezeit nach einem Sende/Empfangen-Zyklus nachdem naechstes mal
		/// Daten gesendet werden
		unsigned int m_nLastSend;
		unsigned int m_nSendInterval;
		
		/// Wartezeit nach der Daten ueber registrierte Spiele auf Server
		/// aktualisiert werdem
		unsigned int m_nLastRefresh;
		unsigned int m_nRefreshInterval;
		
		/// Vektor mit Chat-Nachrichten
		vector<string> m_vszchatMessages;
		/// Maximale Anzahl Nachichten in m_vszchatMessages
		unsigned int m_nMessageQueueLength;

		/// Momentan zu spielende map
		char* m_szCurrentMap;

		/// CGame-Instanz
		CGame* m_pGame;

		/// Die Message-Q verschicken
		/// @return	true wenn Nachrichten gesendet wurden

		int SendMsgQueue();

		/// Nachrichten empfangen und verarbeiten
		/// @return true wenn empfangen werden konnte sonst false

		bool RcvMsgs();

		/// Auf dem Server anmelden (CLIENT_LOGIN)

		void GenClientLogin();

		/// Auf dem Server anmelden (GAME_JOIN)

		void GenGameJoin(u16 nID, const char* szPassword);
		
		/// Ruft Spiele auf dem Server ab (SERVER_GET_GAMES)

		void GenServerGetGames();
		
		/// Ruft Spieler-Informationen ab (SERVER_GET_PLAYER | GAME_GET_PLAYER)

		void GenGetPlayer();
		
		/// Ruft Levelinfomationen ab

		void GenServerGetMaps();
		
		/// Erstellt eine Fortschrittsnachricht für den Server

		void GenProgress();
		
		/// Erstellt eine Zeit-Synchronisations-Nachricht für den Server

		void GenSyncMessage();
		
		/// Auf einem dedicated Server anmelden

		bool DedicatedLogin();
		
		/// Auf einem LAN-Server anmelden

		//bool LANLogin();

		/// Callback fuer CLIENT_LOGIN

		bool HandleClientLogin(u8* pRespBody);

		/// Callback fuer CLIENT_LOGOUT

		bool HandleClientLogout(u8* pRespBody);
		
		/// Callback fuer CLIENT_SYNC

		bool HandleClientSync(u8* pRespBody);

		/// Callback fuer SERVER_GET_GAMES

		bool HandleServerGetGames(u8* pRespBody);

		/// Callback fuer SERVER_CREATE_GAME

		bool HandleServerCreateGame(u8* pRespBody);

		/// Callback fuer GAME_JOIN

		bool HandleGameJoin(u8* pRespBody);

		/// Callback fuer GAME_GET_PLAYER und SERVER_GET_PLAYER

		bool HandleGetPlayer(u8* pRespBody);
		
		/// Callback fuer GAME_GET_PLAYERINFO

		bool HandleGamePlayerInfo(u8* pRespBody);

		/// Callback fuer GAME_START

		bool HandleGameStart(u8* pRespBody);

		/// Callback fuer GAME_GET_MAP

		bool HandleGameGetMap(u8* pRespBody);

		/// Callback fuer GAME_NEXT_LEVEL

		bool HandleGameNextLevel(u8* pRespBody);

		/// Callback fuer GAME_PROGRESS

		bool HandleGameProgress(u8* pRespBody);
		
		/// Fuegt eine Nachricht zur Sende-Warteschlange hinzu
		/// @param	pMessage	CMessage-Objekt

		void AddMessage(CMessage* pMessage);

		/// Blockieren
		/// @see WaitResponse()

		void Block() { m_bBlock = true; }

		/// Blockade loesen
		/// @see WaitResponse()

		void Unblock() { DEBUG_PRINTL("unblock"); m_bBlock = false; }
		
		/// Wartet bis die aktuellen Nachrichten in der Sende-Warteschlange
		/// gesendet und alle Antwortnachrichten verarbeitet wurden

		void WaitResponse();
};

#endif ///CLIENT_HPP_///

