#include "PlayerCtrlMP.hpp"
#include "Game.hpp"
#include "../client/Client.hpp"

CGame::CGame ()
{
	m_bEngineLoaded = false;
	m_pGui = NULL;
	m_pMenu = NULL;
	m_pPlayerCtrl = NULL;
	m_pWorld = NULL;
	m_pClient = NULL;
	m_pLANServerThread = NULL;

	//Start: Menu anzeigen
	m_eProgStatus = MENU_MAIN;
	m_eGameStatus = GAME_STOPPED;

	//Menu initialisieren
	g_pGameMenu->Init(&m_eProgStatus, &m_eGameStatus, this, &m_SaveGame);

	//Level setzen
	m_nLevel = 0;
	m_fLevelTimer = 0;
	m_nSecToEnd = 0;
	memset(m_szLevelFile, 0, 256);

	//Letzten Spielstand laden
	if (!m_SaveGame.Load("Data/Save/save.dat", "Data/Save/highscore.dat"))
		cerr << "Konnte letzten Spielstand nicht laden. Standardwerte gesetzt!" << endl;

	//cout << "SaveGame: Freigeschalten bis Level " << m_SaveGame.GetUnlockedLevel() << endl;

	//Standardwerte
	Reset();
}

CClient* CGame::GetClient()
{
	if (m_pClient == NULL)
	{
		//neu erstellen
		m_pClient = new CClient(this, CLIENT_REFRESH_INTERVAL, CLIENT_SEND_INTERVAL, CLIENT_MSG_QUEUE_LEN);
	}

	return m_pClient;
}

CServer* CGame::GetLANServerThread()
{
	if (m_pLANServerThread == NULL)
	{
		// Neuen Thread erzeugen
		m_pLANServerThread = new CServer(NET_DEFAULT_PORT, false, "Data/Levels/race/");	//Windows braucht den Slash am Ende!
		if (!m_pLANServerThread->Start())
		{
			return false;
		}
	}

	return m_pLANServerThread;
}

void CGame::DestroyClient()
{
	FREEMEM(m_pClient);
}

void CGame::DestroyLANServerThread()
{
	FREEMEM(m_pLANServerThread);
}


void CGame::Quit ()
{
	DestroyClient();
	DestroyLANServerThread();
	SaveGame();
	StopGameEngine();
}

void CGame::Reset()
{
	if (m_bEngineLoaded)
	{
		m_pPlayerCtrl->Reset();
	}

	m_fLevelTimer = 0;
}


SPlayerObj* CGame::GetPlayerObj()
{
	return m_pPlayerCtrl->GetPlayerData();
}

void CGame::SaveGame(bool bUpdateScore)
{
	//Highscore aktualisieren
	if (bUpdateScore && m_pPlayerCtrl && !m_bMPlayer)
	{
		m_pPlayerCtrl->CalcPlayerBonusScore();
		m_SaveGame.UpdateHighscore(m_szLevelFile, m_pPlayerCtrl->GetPlayerData()->nScore + m_pPlayerCtrl->GetPlayerData()->nBonusScore);
	}

	//Abspeichern
	m_SaveGame.Save("Data/Save/save.dat", "Data/Save/highscore.dat");
}

void CGame::StopGameEngine()
{
	if (m_bEngineLoaded)
	{
		FREEMEM(m_pPlayerCtrl);
		FREEMEM(m_pWorld);
		FREEMEM(m_pGui);

		if (m_bMPlayer)
		{
			FREEMEM(m_pClient);
			FREEMEM(m_pLANServerThread);
		}
	}

	m_bEngineLoaded = false;
}

void CGame::LoadGameEngine(bool bMPlayer)
{
	if (m_bEngineLoaded)
		return;

	m_bMPlayer = bMPlayer;

	//Debugfunktionen:
	m_bDrawAxis = false;

	//PlayerControll Objekt laden
	if (bMPlayer)
		m_pPlayerCtrl = new CPlayerCtrlMP(GetClient(), &m_eProgStatus, &m_eGameStatus);
	else
		m_pPlayerCtrl = new CPlayerCtrl(&m_eProgStatus, &m_eGameStatus);

	//GUI laden
	m_pGui = new CGui(this, m_pPlayerCtrl->GetPlayerData());
	m_pPlayerCtrl->SetGuiPointer(m_pGui);

	//Level-Objekt laden
	m_pWorld = new CWorld;
	m_pWorld->SetPlayer(m_pPlayerCtrl->GetPlayerData());
	m_pPlayerCtrl->SetWorldPointer(m_pWorld);

	//Particle-Engine laden
	g_pParticle->SetWorldPointer(m_pWorld);

	m_nSecToEnd = 300;	// 3 Sek. warten am Ende des Spiels

	//Erfolgreich geladen
	m_bEngineLoaded = true;
}

bool CGame::LoadLevel()
{
	cout << "Lade Level " << m_szLevelFile << "..." << endl;

	if (m_bMPlayer)	//den Index fuer den Startblock mitteilen
	{
		m_pWorld->SetPlayerStartPos(GetClient()->GetPlayerStartPos());
	}

	if (!m_pWorld->LoadFromFile(m_szLevelFile))
	{
		cerr << "Could not load Level-File: " << m_szLevelFile << endl;
		m_eProgStatus = PROGRAM_EXIT;

		return false;
	}

	return true;
}

bool CGame::LoadLevel(const int nLevel)
{
	m_nLevel = nLevel;

	if (!m_bMPlayer)
		sprintf(m_szLevelFile, "Data/Levels/taxi/taxi%d.txt", nLevel);
	else
		sprintf(m_szLevelFile, "Data/Levels/race/race%d.txt", nLevel);

	return LoadLevel();
}

bool CGame::LoadLevel(const char* szLevelFile)
{
	strcpy(m_szLevelFile, szLevelFile);

	return LoadLevel();
}

bool CGame::LoadNextLevel()
{
	//Level raufz�hlen
	m_nLevel++;

	//H�chstes Level erreicht!
	if (m_nLevel > GAME_MAX_LEVEL)
	{
		return false;
	}

	//Neues Level freischalten!
	if (m_SaveGame.GetSaveDataPointer()->nUnlockedLevel < m_nLevel)
		m_SaveGame.GetSaveDataPointer()->nUnlockedLevel = m_nLevel;

	//Leveldatei einlesen
	return LoadLevel(m_nLevel);
}

SSaveData* CGame::GetSavedPlayerData()
{
	return m_SaveGame.GetSaveDataPointer();
}

void CGame::Run ()
{
	// Hauptschleife des Spiels durchlaufen
	while (m_eProgStatus != PROGRAM_EXIT)
	{
		g_pFramework->Update();

		if (m_eProgStatus == INGAME)	//In-game: Spiel darstellen
		{
			if (CGameMenu::IsInstantiated())
				g_pGameMenu->Del();

			ProcessEvents();
			DrawScene();
		}
		else	//Sonst muessen wir wohl ein Menu darstellen
		{
			if (!CGameMenu::IsInstantiated())
			{
				g_pGameMenu->Init(&m_eProgStatus, &m_eGameStatus, this, &m_SaveGame);
			}

			g_pGameMenu->Draw();
			g_pGameMenu->Update();
		}

		// Fenster geschlossen?
		SDL_Event* pEvent = g_pFramework->GetEvent();
		if (pEvent != NULL && pEvent->type == SDL_QUIT)
		{
			m_eProgStatus = PROGRAM_EXIT;
		}
		else
		{	// Buffer flippen
			g_pFramework->Flip();
		}
	}
}


void CGame::DrawScene()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);	// Clear Screen And Depth Buffer
	glLoadIdentity();									// Reset The Current Modelview Matrix

	//Nach Spielstatus gucken im MP-Modus
	if (m_bMPlayer)
	{
		if (m_pClient->GetRunningGame()->eState == SERVERGAME_END)
		{
			if (m_nSecToEnd > 0)
			{
				m_pGui->Update();
				char szWaitMessage[256];
				sprintf(szWaitMessage, "Rennen endet in %d Sekunden", (int)(m_nSecToEnd / 100));

				C2DTools::Enter2DMode();
				g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.9, 0.0);
				g_pText->DrawStr(300, 300, szWaitMessage);
				C2DTools::Leave2DMode();

				m_nSecToEnd -= g_pTimer->GetElapsed();
				return;
			}
			else
			{
				m_eProgStatus = MENU_MPLAYER_ENDSTATS;
				m_eGameStatus = GAME_PAUSED;
			}
			//return;
		}
		else if (m_nLevel != m_pClient->GetRunningGame()->nCurrentLevel)	//Levelwechsel!
		{
			Reset();
			SetCurrentLevel(m_pClient->GetRunningGame()->nCurrentLevel);
			if (!LoadLevel(m_pClient->GetMapFile()))
			{
				m_eProgStatus =  PROGRAM_EXIT;
				return;
			}
		}
		else if (m_pClient->LevelRunning() == false)
		{
			char szWaitMessage[256];
			sprintf(szWaitMessage, "Rennen startet in %d Sekunden", m_pClient->SecondsToStart());

			C2DTools::Enter2DMode();
			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.9, 0.0);
			g_pText->DrawStr(300, 300, szWaitMessage);
			C2DTools::Leave2DMode();
			return;
		}
	}

	//Welt zeichnen
	m_pWorld->Update();
	m_pWorld->Draw();

	if (m_bDrawAxis)
		C3DTools::drawAxis(2.0);

	//Spieler zeichnen
	m_pPlayerCtrl->Draw(m_bDrawAxis);
	m_pPlayerCtrl->Update();

	/*SPlayerObj* pPlayerData = m_pPlayerCtrl->GetPlayerData();
	cout << pPlayerData->fPosX << ", " << pPlayerData->fPosY << ", " << pPlayerData->fPosZ << endl;*/

	//Transparente Objekte der Welt als vorletztes zeichnen
	m_pWorld->DrawTransparentObjects();

	//GUI malen
	m_pGui->Draw();

	//Leveltimer hochz�hlen
	m_fLevelTimer += g_pTimer->GetElapsed();
}

void CGame::ProcessEvents()
{
	SDL_Event* pEvent = g_pFramework->GetEvent();

	// Gab es ein Event?
	//if (SDL_PollEvent(&Event))
	if (pEvent != NULL)
	{
		// Ja, also schauen welches
		switch (pEvent->type)
		{
			// Wurde eine Taste gedr�ckt?
			case (SDL_KEYDOWN):
				switch (pEvent->key.keysym.sym)
				{
					case (SDLK_ESCAPE):
						//ESC -> Hauptmenu!
						m_eProgStatus = MENU_PAUSE;
						m_eGameStatus = GAME_PAUSED;
					break;

					// F1: Hilfe
					case (SDLK_F1):
						m_eProgStatus = MENU_HELP;
						m_eGameStatus = GAME_PAUSED;
					break;

#ifdef _DEBUG
					// F5: Neuladen des Levels aus der Datei
					case (SDLK_F5):
						m_pWorld->Reload();
					break;

					// F10: Koordinatensys. zeichnen
					case (SDLK_F10):
						m_bDrawAxis = !m_bDrawAxis;
					break;
#endif
					default:
						break;
				}
			break;

			default:
				break;
        }
	}
}
