#ifndef GAME_HPP
#define GAME_HPP

#include "../client/PlatformFramework.hpp"

#include "../globals.hpp"
#include "../client/Framework.hpp"
#include "../server/Server.hpp"
#include "../tools/3DTools.hpp"
#include "../gfx/Texture.hpp"
#include "../gfx/Particle.hpp"
#include "../gfx/Gui.hpp"

#include "PlayerCtrl.hpp"
#include "World.hpp"
#include "GameMenu.hpp"
#include "SaveGame.hpp"

class CClient;
class CGameMenu;
class CServer;
class CPlayerCtrl;
class CWorld;
class CGui;

/// Hauptklasse fuer den Spielablauf
class CGame
{
	public:
		/// Konstruktor: Uebernimmt Initialisierung des Spiels
		CGame();

		/// Hauptschleife des Spiels
		/// Schleife wird bis zum beenden des Spiels durchlaufen. Dabei
		/// werden Unterfunktionen aufgerufen.
		void Run();

		/// Spiel beenden und Speicher freigeben
		/// Achtung! JEDER auf dem Heap erzeugte Speicher muss
		/// hier freigegeben werden!
		void Quit();

		/// Alles zurueck setzen
		void Reset();

		/// Gibt ein CClient-Objekt zurueck und erzeugt eventuell eines
		/// @return Zeiger auf CClient-Objekt
		CClient* GetClient();

		/// Gibt ein CServer-Objekt zurueck und erzeugt eventuell eines
		/// @return Zeiger auf CServer-Objekt
		CServer* GetLANServerThread();
		
		/// Zerstoert das CClient-Objekt
		void DestroyClient();

		/// Zerstoert das CServer-Objekt
		void DestroyLANServerThread();

		/// Spielstand speichern
		/// @param bUpdateScore Spielstand aktualisieren?
		void SaveGame(bool bUpdateScore = true);

		/// Neue Instanzen fuer das Spiel erzeugen
		/// @param bMPlayer MP-Modus starten?
		void LoadGameEngine(bool bMPlayer = false);

		/// Speicher frei geben
		void StopGameEngine();

		/// Multiplayer-Spiel?
		/// @return Multiplayer-Spiel true/false
		bool IsMPlayer() { return m_bMPlayer; }

		/// Aktuelles Level laden
		bool LoadLevel();

		/// Bestimmtes Level laden
		bool LoadLevel(const int nLevel);

		/// Bestimmte Leveldatei laden
		bool LoadLevel(const char* szLevelFile);

		/// Naechstes Level laden
		/// @return bool Gibt es ein naechstes Level? Bei false wurden alle Levels durchgespielt.
		bool LoadNextLevel();

		/// Aktuelles Level zurueckgeben
		/// @return Aktuelles Level
		int GetCurrentLevel() { return m_nLevel; };

		/// Spielerdaten zurueckgeben
		/// @return Zeiger auf Spielerdaten
		SPlayerObj* GetPlayerObj();

		/// Gespeicherte Spielereinstellungen zurueckgeben
		/// @return Zeiger auf gespeicherte Spielereinstellungen
		SSaveData* GetSavedPlayerData();

		/// GUI-Objekt zurueck geben
		/// @return Zeiger auf GUI
		CGui* GetGui() { return m_pGui; }

		/// Aktuelles Level setzen
		/// @param nLevel Level-Nummer
		void SetCurrentLevel(int nLevel) { m_nLevel = nLevel; m_szLevelFile[0] = '\0'; };

		/// Leveltimer mit Zeit in Sekunden zurueckgeben
		/// @return Spielzeit in diesem Level
		float GetLevelTimer() { return m_fLevelTimer; };

	private:
		/// 3D-Szene abbilden
		/// Hier werden die einzelnen Objekte gerendert. Konkret heisst das,
		/// das die Draw() Funktionen der Objekte, wie z.B. m_pPlayer oder m_pWorld
		/// aufgerufen werden.
		void DrawScene();

		/// Events abfangen und darauf reagieren.
		/// Hiermit sind v.a. Tastendruecke gemeint, allerdings "globale" Tasten,
		/// d.h. die Tasten, die nicht direkt ein Objekt im Spiel steuern, sondern
		/// z.B. das Spiel beenden, die Pause aufrufen oder Debugfunktionen aktivieren
		/// Tasten, die bestimmte Objekte steuern, z.B. den Spieler, muessen direkt in der
		/// jeweiligen ProcessEvents() oder Update() Funktion dieses Objekts definiert
		/// werden.
		void ProcessEvents();
		
		/// Zurueck auf Ausgangszustand nach Tod
		void ResetAfterDeath(const int nDeathBlockX, const int nDeathBlockY, const int nDeathBlockZ);


		float m_fLevelTimer;			///< Timer fuer die Level-Uhr und fuer die Bonuspunkteberechnung spaeter
		bool m_bDrawAxis;				///< Debug: Koordinatensysteme zeichnen?
		bool m_bEngineLoaded;			///< Engine geladen, d.h. noetige Klasseninstanzen erzeugt?

		int m_nLevel;					///< derzeitiges Level
		char m_szLevelFile[256];		///< akt. Leveldatei
		int m_nSecToEnd;				///< Timer bis zum Ende des Spiels

		CPlayerCtrl* m_pPlayerCtrl;		///< PlayerCtrl-Objekt
		CWorld*  m_pWorld;				///< Levelinstanz
		CGui*  m_pGui;					///< GUI-Instanz
		CGameMenu* m_pMenu;				///< Menu-Instanz
		CClient* m_pClient;				///< Client-Instanz beim MP-Spiel
		CServer* m_pLANServerThread;	///< Fuer den Custom-Server existiert hier ein Zeiger auf den Server-Thread

		CSaveGame m_SaveGame;			///< Gespeicherte Daten
		EGameStatus m_eGameStatus;		///< Spielstatus
		EProgramStatus m_eProgStatus;	///< Programmstatus
		bool m_bMPlayer;				///< Multiplayermodus?
};

#endif
