#include "GameMenu.hpp"
#include "../client/Client.hpp"
#include "../gfx/Gui.hpp"

CGameMenu::CGameMenu()
{
	m_nRegisterTyping = -1;
	m_nTypingTextMaxChars = 0;

	m_bConnectionError = false;
	m_bJoinError = false;
	m_bIPEntered = false;
	m_fConnectionStarted = 0.0;

	m_nSelectedMenuTitle	= 0;
	m_fSelectedColorIntensity = 0.0;

	m_pPlayerObj = new SPlayerObj;
	m_pPlayerModel = new CPlayer(m_pPlayerObj);
	m_pPlayerModel->SetPlayerPos(0.0, 0.7, -3.0);
	m_pPlayerModel->SetPlayerAngle(0.0, 0.0, 0.0);

	m_nFadeInFrames = 0;
	m_nSelectedGame = 0;
}

CGameMenu::~CGameMenu(void)
{
	FREEMEM(m_pPlayerModel);
	FREEMEM(m_pPlayerObj);
}

void CGameMenu::Init(EProgramStatus* peProgStatus, EGameStatus* peGameStatus, CGame* pGame, CSaveGame* pSaveGame)
{
	m_peProgStatus = peProgStatus;
	m_peGameStatus = peGameStatus;
	m_pGame = pGame;
	m_pSaveGame = pSaveGame;
}

void CGameMenu::Update()
{
	m_fSelectedColorIntensity = sinf(g_pTimer->GetCurTime() * 5.0) * 0.25 + 0.75;	//Sinus-Funktion f�r den Wechsel der Helligkeit bei aktueller Men�auswahl

	if (m_nRegisterTyping > -1)
	{
		ProcessTyping();
	}
	else
	{
		ProcessEvents();
	}

	m_vMenuTitles.clear();
}

void CGameMenu::ProcessEvents()
{
	SDL_Event* pEvent = g_pFramework->GetEvent();

	// Gab es ein Event?
	//if (SDL_PollEvent (&Event))
	if (pEvent != NULL)
	{
		// Ja, also schauen welches
		switch (pEvent->type)
		{
			// Wurde eine Taste gedr�ckt?
			case (SDL_KEYDOWN):
				switch (pEvent->key.keysym.sym)
				{
					case (SDLK_ESCAPE):
						//ESC -> Programm Beenden oder zur�ck!
						switch (*m_peProgStatus)
						{
							case MENU_PAUSE:
								m_nFadeInFrames = 0;
								*m_peProgStatus	= INGAME;
								*m_peGameStatus	= GAME_RUNNING;
							break;

							case MENU_MAIN:
								*m_peProgStatus	= PROGRAM_EXIT;
							break;

							case MENU_LEVEL:
							case MENU_HELP:
							case MENU_CREDITS:
								if (*m_peProgStatus == MENU_HELP && *m_peGameStatus == GAME_PAUSED)
								{
									m_nFadeInFrames = 0;
									*m_peProgStatus	= INGAME;
									*m_peGameStatus	= GAME_RUNNING;
								}
								else
									*m_peProgStatus	= MENU_MAIN;
							break;
						}
					break;

					case (SDLK_UP):
						m_nSelectedMenuTitle--;

						if (m_nSelectedMenuTitle < 0)
							m_nSelectedMenuTitle = m_vMenuTitles.size() - 1;
					break;

					case (SDLK_DOWN):
						m_nSelectedMenuTitle++;

						if (m_nSelectedMenuTitle > (int)m_vMenuTitles.size() - 1)
							m_nSelectedMenuTitle = 0;
					break;

					case (SDLK_RETURN):
						ProcessMenuSelect();
					break;
				}
			break;
        }
	}
}

void CGameMenu::EnterNewMode(EProgramStatus eProgStatus, EGameStatus eGameStatus)
{
	m_nFadeInFrames			= 0;
	m_nSelectedMenuTitle	= 0;

	*m_peProgStatus = eProgStatus;
	*m_peGameStatus = eGameStatus;
}

void CGameMenu::StartTyping(void (CGameMenu::*pfInputHandler)(void), int nMaxChars, int nInputField)
{
	m_nRegisterTyping = nInputField;
	m_nTypingTextMaxChars = nMaxChars;
	m_pfInputHandler = pfInputHandler;
}

void CGameMenu::ProcessTyping()
{
	//m_bRegisterTyping = true;
	SDL_Event* pEvent = g_pFramework->GetEvent();
	char* szPressedKey;
	char chKey;

	// Gab es ein Event?
	if (pEvent != NULL)
	{
		if (pEvent->type == SDL_KEYDOWN)
		{
			if (pEvent->key.keysym.sym == SDLK_RETURN
				|| pEvent->key.keysym.sym == SDLK_ESCAPE)
			{
				if (m_pfInputHandler)
				{
					((*this).*m_pfInputHandler)();
				}

				m_nRegisterTyping = -1;
				return;
			}
			else if (pEvent->key.keysym.sym == SDLK_BACKSPACE && m_vsInput[m_nRegisterTyping].length() > 0)
			{
				m_vsInput[m_nRegisterTyping] = m_vsInput[m_nRegisterTyping].erase(m_vsInput[m_nRegisterTyping].length() - 1, 1);
			}
			else if (m_vsInput[m_nRegisterTyping].length() < m_nTypingTextMaxChars)
			{
				szPressedKey = SDL_GetKeyName(pEvent->key.keysym.sym);

				if (strlen(szPressedKey) == 1)
				{
					m_vsInput[m_nRegisterTyping] += szPressedKey;
				}
			}
		}
	}
}

void CGameMenu::HandlePlayerNameInput()
{
	CFileTools::StrToUpper(m_vsInput[m_nRegisterTyping]);
	m_pSaveGame->GetSaveDataPointer()->sPlayerName = m_vsInput[m_nRegisterTyping];
}


void CGameMenu::HandleIPInput()
{
	DEBUG_PRINTL("Entered IP: " << m_vsInput[m_nRegisterTyping]);

	m_sEnteredIP = m_vsInput[m_nRegisterTyping];
	m_vAvailableGames.clear();	//Wichtig!

	m_bIPEntered = true;

	//Verbinden...
	if (*m_peProgStatus == MENU_MPLAYER_JOIN)
	{
		m_pGame->DestroyClient();
		CClient* pClient = m_pGame->GetClient();

		if (pClient->connect(m_vsInput[m_nRegisterTyping].c_str(), NET_DEFAULT_PORT))
		{
			m_bConnectionError = false;
			m_fConnectionStarted = SDL_GetTicks();
		}
		else
		{
			m_bConnectionError = true;
			DEBUG_PRINTL("connection refused to server " << m_vsInput[m_nRegisterTyping] << " on port " << NET_DEFAULT_PORT);
			m_pGame->DestroyClient();
		}
	}
}

void CGameMenu::ProcessMenuSelect()
{
	switch (*m_peProgStatus)
	{
		//Im Hauptmenu / Pausemenu
		case MENU_MAIN:
			switch (m_nSelectedMenuTitle)
			{
				//Start
				case 0:
					m_pGame->LoadGameEngine();
					m_pGame->Reset();

					if (!m_pGame->LoadNextLevel() && *m_peProgStatus == PROGRAM_EXIT)
					{
						//etwas ist schief gegangen beim Level laden
						break;
					}

					EnterNewMode(INGAME, GAME_RUNNING);		//Spiel starten
				break;

				//Levelselect
				case 1:
					EnterNewMode(MENU_LEVEL, *m_peGameStatus);
				break;

				//Multiplayer
				case 2:
					EnterNewMode(MENU_MPLAYER, *m_peGameStatus);
				break;

				//Hilfe
				case 3:
					EnterNewMode(MENU_HELP, *m_peGameStatus);
				break;

				//Credits
				case 4:
					EnterNewMode(MENU_CREDITS, *m_peGameStatus);
				break;

				//Beenden
				case 5:
					EnterNewMode(PROGRAM_EXIT, *m_peGameStatus);
				break;
			}
		break;

		//Pause Men�
		case MENU_PAUSE:
			switch (m_nSelectedMenuTitle)
			{
				//Weiter
				case 0:
					EnterNewMode(INGAME, GAME_RUNNING);		//Fortfahren
				break;

				//Zur Hilfe
				/*case 1:
					EnterNewMode(MENU_HELP, *m_peGameStatus);
				break;*/

				//Zum Hauptmenu
				case 1:
					EnterNewMode(MENU_MAIN, GAME_STOPPED);
					m_pGame->SaveGame(false);
					m_pGame->StopGameEngine();
					m_pGame->SetCurrentLevel(0);
				break;

				//Beenden
				case 2:
					EnterNewMode(PROGRAM_EXIT, *m_peGameStatus);
					m_pGame->SaveGame(false);
					m_pGame->StopGameEngine();
					m_pGame->SetCurrentLevel(0);
				break;
			}
		break;


		//Hilfe
		case MENU_HELP:
			switch (m_nSelectedMenuTitle)
			{
				//Zur�ck zum Hauptmenu oder zum Spiel
				case 0:
					if (*m_peGameStatus == GAME_STOPPED)
						EnterNewMode(MENU_MAIN, *m_peGameStatus);
					else
						EnterNewMode(INGAME, *m_peGameStatus);
				break;
			}
		break;


		//Im Creditsmenu
		case MENU_CREDITS:
			switch (m_nSelectedMenuTitle)
			{
				//Zur�ck zum Hauptmenu
				case 0:
					EnterNewMode(MENU_MAIN, *m_peGameStatus);
				break;
			}
		break;

		//Levelende
		case MENU_LEVEL_END:
			switch (m_nSelectedMenuTitle)
			{
				//N�chstes Level laden!
				case 0:
					m_pGame->SaveGame();
					m_pGame->Reset();

					if (!m_pGame->LoadNextLevel())
					{
						//Es gibt kein n�chstes Level! H�chstes Level erreicht!
						EnterNewMode(MENU_GAME_COMPLETED, GAME_STOPPED);		//Fortfahren
						//m_pGame->StopGameEngine();
					}
					else
					{
						EnterNewMode(INGAME, GAME_RUNNING);		//Fortfahren
					}
				break;
			}
		break;

		//Spiel geschafft oder Game Over
		case MENU_GAME_COMPLETED:
			m_pGame->SaveGame();
			m_pGame->StopGameEngine();
		case MENU_GAME_OVER:
			switch (m_nSelectedMenuTitle)
			{
				//Zur�ck zum Hauptmen�
				case 0:
					m_pGame->SetCurrentLevel(0);
					EnterNewMode(MENU_MAIN, GAME_STOPPED);
				break;
			}
		break;

		//Levelauswahl
		case MENU_LEVEL:
			if (m_nSelectedMenuTitle == (int)m_vMenuTitles.size() - 1)	//letzter Men�eintrag -> Zur�ck
			{
				EnterNewMode(MENU_MAIN, GAME_STOPPED);
			}
			else
			{
				m_pGame->LoadGameEngine();
				m_pGame->Reset();

				if (!m_pGame->LoadLevel(m_nSelectedMenuTitle + 1) && *m_peProgStatus == PROGRAM_EXIT)
				{
					//etwas ist schief gegangen beim Level laden
					break;
				}

				EnterNewMode(INGAME, GAME_RUNNING);
			}
		break;

		//Multiplayer
		case MENU_MPLAYER:
			switch (m_nSelectedMenuTitle)
			{
				//Player-Setup
				case 0:
					m_vsInput.clear();
					m_vsInput.push_back(m_pSaveGame->GetSaveDataPointer()->sPlayerName);
					EnterNewMode(MENU_MPLAYER_PLAYERSETUP, *m_peGameStatus);
				break;

				//Neues LAN-Spiel erstellen
				case 1:
					m_vSelectedMaps.clear();
					m_bIPEntered = false;	// bei LAN-Spiel immer false, bei Internetspiel erstellen true!
					EnterNewMode(MENU_MPLAYER_CREATE, *m_peGameStatus);
				break;

				//Neues Internet-Spiel erstellen
				case 2:
					m_vSelectedMaps.clear();
					m_vsInput.push_back("");
					m_vAvailableGames.clear();
					EnterNewMode(MENU_MPLAYER_CREATE_INET, *m_peGameStatus);
				break;

				//Spiel beitreten
				case 3:
					m_vsInput.clear();
					m_vsInput.push_back("");
					m_vAvailableGames.clear();
					EnterNewMode(MENU_MPLAYER_JOIN, *m_peGameStatus);
				break;

				//Zurueck zum Hauptmenue
				case 4:
					m_pGame->SetCurrentLevel(0);
					EnterNewMode(MENU_MAIN, GAME_STOPPED);
				break;
			}
		break;

		//Multiplayer - Spiel erstellen
		case MENU_MPLAYER_CREATE:
			if (m_nSelectedMenuTitle == (int)m_vMenuTitles.size() - 1)	//letzter Menueeintrag -> Zurueck
			{
				EnterNewMode(MENU_MPLAYER, GAME_STOPPED);
			}
			else if (m_nSelectedMenuTitle == (int)m_vMenuTitles.size() - 2 && m_vSelectedMaps.size() > 0)	//vorletzter Menueeintrag -> Weiter
			{
				m_vsInput.clear();
				m_vsInput.push_back("testspiel");
				m_vsInput.push_back("2");
				m_vsInput.push_back("\0");
				m_bConnectionError = false;
				EnterNewMode(MENU_MPLAYER_CREATECONFIG, GAME_STOPPED);
			}
			else	//eine Map wurde ausgewaehlt -> im Vektor speichern!
			{
				m_vSelectedMaps.push_back((u8)m_nSelectedMenuTitle);
			}
		break;

		//Multiplayer - Config fuer den Custom Server
		case MENU_MPLAYER_CREATECONFIG:
			m_pGame->DestroyLANServerThread();
			switch (m_nSelectedMenuTitle)
			{
				//Spielnamen eintippen
				case 0:
					StartTyping(NULL, GAME_NAME_LENGTH, 0);
				break;

				//Max. Spieler eintippen
				case 1:
					StartTyping(NULL, 2, 1);
				break;

				//Passwort eintippen
				case 2:
					StartTyping(NULL, GAME_PASSWD_LENGTH, 2);
				break;

				//Weiter
				case 3:
				{
					g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.9, 0.0);
					g_pText->DrawStr(520, 352, "Starte Server...");

					CClient* pClient = m_pGame->GetClient();
					int nTimeout = 0;

					if (!m_bIPEntered)
					{
						//erstellt einen neuen LAN-Server-Thread
						m_pLANServerThread = m_pGame->GetLANServerThread();

						while (m_pLANServerThread->IsRunning() == false && nTimeout <= 10)	//warten bis der Server wirklich gestartet ist
						{
							DEBUG_PRINTL("Starte LAN-Server-Thread...");
							SDL_Delay(100);
							nTimeout++;
						}

						if (nTimeout > 10)
						{
							DEBUG_PRINTL("LAN-Server-Thread start failed");
							m_pGame->DestroyClient();
							m_pGame->DestroyLANServerThread();
							m_bConnectionError = true;
							break;
						}

						//Loopback-Verbindung aufbauen zum Server
						pClient->connect("127.0.0.1", NET_DEFAULT_PORT);
					}
					else
					{
						//Verbindung aufbauen zum Dedi-Server
						DEBUG_PRINTL("Verbinde zum Server " << m_sEnteredIP);
						pClient->connect(m_sEnteredIP.c_str(), NET_DEFAULT_PORT);
					}

					nTimeout = 0;
					while (pClient->GetLocalPlayerID() == 0 && nTimeout <= 10)	//warten bis wir wirklich verbunden und eingeloggt sind
					{
						DEBUG_PRINTL("Connection failed");
						SDL_Delay(100);
						nTimeout++;
					}

					if (nTimeout > 10)
					{
						m_pGame->DestroyClient();
						m_pGame->DestroyLANServerThread();
						m_bConnectionError = true;
						break;
					}

					//Spiel erstellen
					m_pGame->GetClient()->CreateGame(m_vsInput[0].c_str(), atoi(m_vsInput[1].c_str()), true, true, m_vsInput[2].c_str(), m_vSelectedMaps);

					nTimeout = 0;
					while (pClient->GetCurrGameID() < 0 && nTimeout <= 10)	//warten bis wir wirklich das Spiel erstellt haben
					{
						SDL_Delay(100);
					}

					if (nTimeout > 10)
					{
						m_pGame->DestroyClient();
						m_pGame->DestroyLANServerThread();
						m_bConnectionError = true;
						break;
					}

					//Spiel Infos holen
					m_nSelectedGame = -1;
					nTimeout = 0;
					while (m_nSelectedGame < 0 && nTimeout <= 10)
					{
						m_vAvailableGames = m_pGame->GetClient()->GetGames();
						for (int i = 0; i < m_vAvailableGames.size(); i++)
						{
							if (m_vAvailableGames[i].nID == pClient->GetCurrGameID())
							{
								m_nSelectedGame = i;
							}
						}
						SDL_Delay(100);
						nTimeout++;
					}

					if (nTimeout > 10)
					{
						m_pGame->DestroyClient();
						m_pGame->DestroyLANServerThread();
						m_bConnectionError = true;
						break;
					}

					EnterNewMode(MENU_MPLAYER_GAMEROOM, *m_peGameStatus);
				}
				break;

				//Zurueck
				case 4:
					EnterNewMode(MENU_MPLAYER_CREATE, *m_peGameStatus);
				break;
			}
		break;

		//Multiplayer - Spieler Einstellungen
		case MENU_MPLAYER_PLAYERSETUP:
			switch (m_nSelectedMenuTitle)
			{
				//Name eintippen
				case 0:
					StartTyping(&CGameMenu::HandlePlayerNameInput, PLAYER_NAME_LENGTH, 0);
				break;

				//Zur�ck
				case 1:
					EnterNewMode(MENU_MPLAYER, *m_peGameStatus);
				break;
			}
		break;

		//Multiplayer - Internet Spiel erstellen
		case MENU_MPLAYER_CREATE_INET:
			if (m_nSelectedMenuTitle == 0)	//IP eintippen
			{
				StartTyping(&CGameMenu::HandleIPInput, 15, 0);
			}
			else if (m_nSelectedMenuTitle == m_vMenuTitles.size() - 1)	//Zurueck
			{
				//m_pGame->DestroyClient();
				m_bIPEntered = false;
				EnterNewMode(MENU_MPLAYER, *m_peGameStatus);
			}
			else	//Server ausgewaehlt!
			{
				EnterNewMode(MENU_MPLAYER_CREATE, *m_peGameStatus);
			}
		break;

		//Multiplayer - Spiel beitreten
		case MENU_MPLAYER_JOIN:
			if (m_nSelectedMenuTitle == 0)	//IP eintippen
			{
				StartTyping(&CGameMenu::HandleIPInput, 15, 0);
			}
			else if (m_nSelectedMenuTitle == m_vAvailableGames.size() + 1)	//Zurueck
			{
				m_pGame->DestroyClient();
				m_bIPEntered = false;
				EnterNewMode(MENU_MPLAYER, *m_peGameStatus);
			}
			else	//Spiel ausgewaehlt!
			{
				m_nSelectedGame = m_nSelectedMenuTitle - 1;

				if (m_vAvailableGames[m_nSelectedGame].bUsePassword)
				{
					m_vsInput.clear();
					m_vsInput.push_back("\0");
					EnterNewMode(MENU_MPLAYER_JOIN_PASSWD, *m_peGameStatus);
				}
				else
				{
					//beitreten...
					ActionJoin(NULL);
				}
			}
		break;

		//Multiplayer - Passwort eingeben um Spiel beitreten zu koennen
		case MENU_MPLAYER_JOIN_PASSWD:
			switch (m_nSelectedMenuTitle)
			{
				//Passwort eingeben
				case 0:
					StartTyping(NULL, 15, 0);
				break;

				//Spiel beitreten
				case 1:
					ActionJoin(m_vsInput[0].c_str());
				break;

				//Zurueck
				case 2:
					m_vsInput.clear();
					EnterNewMode(MENU_MPLAYER_JOIN, *m_peGameStatus);
				break;
			}
		break;

		//Multiplayer - Im Spielvorbereitungsraum
		case MENU_MPLAYER_GAMEROOM:
			if (m_nSelectedMenuTitle == 0)	//Zurueck
			{
				m_pGame->GetClient()->LeaveGame();
				m_pGame->DestroyLANServerThread();
				m_vAvailableGames.clear();
				m_bIPEntered = false;
				EnterNewMode(MENU_MPLAYER_JOIN, *m_peGameStatus);
			}
			else	//Spieler auf bereit setzen
			{
				m_pGame->GetClient()->StartGame();
			}
		break;

		//Multiplayer - Endstats nach dem Spiel
		case MENU_MPLAYER_ENDSTATS:
			if (m_nSelectedMenuTitle == 0)	//OK
			{
				m_pGame->GetClient()->LeaveGame();
				m_pGame->StopGameEngine();
				m_vAvailableGames.clear();
				m_bIPEntered = false;
				EnterNewMode(MENU_MPLAYER, GAME_STOPPED);
			}
		break;
	}
}

void CGameMenu::ActionJoin(const char* szPasswd)
{
	m_bJoinError = false;
	m_pGame->GetClient()->JoinGame(m_vAvailableGames[m_nSelectedGame].nID, szPasswd);

	if (!m_pGame->GetClient()->InGame())	//Fehler beim Beitreten
	{
		m_bJoinError = true;
		EnterNewMode(MENU_MPLAYER_JOIN, *m_peGameStatus);
	}
	else
	{
		//Alles klaro, also ab ins Spielzimmer!
		EnterNewMode(MENU_MPLAYER_GAMEROOM, *m_peGameStatus);
	}
}

void CGameMenu::ActionStartMPGame()
{
	m_pGame->LoadGameEngine(true);
	m_pGame->Reset();
	m_pGame->SetCurrentLevel(m_pGame->GetClient()->GetRunningGame()->nCurrentLevel);

	if (m_pGame->LoadLevel(m_pGame->GetClient()->GetMapFile()))
		EnterNewMode(INGAME, GAME_RUNNING);
	else
		EnterNewMode(PROGRAM_EXIT, GAME_RUNNING);
}

void CGameMenu::Draw()
{
	switch (*m_peProgStatus)
	{
		case MENU_MAIN:
			DrawDefaultMenu("TRAXI 2.0", MENUID_MAIN);
		break;

		case MENU_PAUSE:
			DrawDefaultMenu("TRAXI 2.0 | PAUSE", MENUID_GAME_PAUSED, 200, 250, false);
		break;

		case MENU_HELP:
			if (*m_peGameStatus == GAME_STOPPED)
			{
				ResetView();
				DrawMessageBox("Hilfe", TEXTID_HELP, MENUID_BACK, true, false);
			}
			else
			{
				DrawMessageBox("Hilfe", TEXTID_HELP, MENUID_BACK);
			}
		break;

		case MENU_LEVEL:
			DrawDefaultMenu("TRAXI 2.0 | LEVELAUSWAHL", MENUID_LEVELS, 200, 290);
		break;

		case MENU_CREDITS:
			ResetView();
			DrawMessageBox("Credits", TEXTID_CREDITS, MENUID_BACK, true, false);
		break;

		case MENU_LEVEL_END:
			DrawMessageBox("Mission erfolgreich!", TEXTID_LEVEL_END, MENUID_OK);
		break;

		case MENU_GAME_OVER:
			ResetView();
			DrawMessageBox("Game Over :(", TEXTID_GAME_OVER, MENUID_OK, true, false);
		break;

		case MENU_GAME_COMPLETED:
			ResetView();
			DrawMessageBox("GESCHAFFT!", TEXTID_GAME_COMPLETED, MENUID_OK, true, false);
		break;

		case MENU_MPLAYER:
			DrawDefaultMenu("TRAXI 2.0 | MULTIPLAYER", MENUID_MPLAYER);
		break;

		case MENU_MPLAYER_PLAYERSETUP:
			DrawDefaultMenu("TRAXI 2.0 | SPIELEREINSTELLUNGEN", MENUID_MPLAYER_PLAYERSETUP);
		break;

		case MENU_MPLAYER_CREATE:
			DrawDefaultMenu("TRAXI 2.0 | SPIEL ERSTELLEN - LEVELAUSWAHL", MENUID_MPLAYER_CREATE, 200, 290);
		break;

		case MENU_MPLAYER_CREATE_INET:
			DrawDefaultMenu("TRAXI 2.0 | INTERNET SPIEL ERSTELLEN", MENUID_MPLAYER_CREATE_INET);
		break;

		case MENU_MPLAYER_CREATECONFIG:
			DrawDefaultMenu("TRAXI 2.0 | SPIEL ERSTELLEN - OPTIONEN", MENUID_MPLAYER_CREATECONFIG);
		break;

		case MENU_MPLAYER_JOIN:
			DrawDefaultMenu("TRAXI 2.0 | SPIEL BEITRETEN", MENUID_MPLAYER_JOIN);
		break;

		case MENU_MPLAYER_JOIN_PASSWD:
			DrawDefaultMenu("TRAXI 2.0 | SPIEL PASSWORT EINGEBEN", MENUID_MPLAYER_JOIN_PASSWD);
		break;

		case MENU_MPLAYER_GAMEROOM:
			DrawGameRoom();
		break;

		case MENU_MPLAYER_ENDSTATS:
			DrawDefaultMenu("TRAXI 2.0 | SPIELSTATISTIKEN", MENUID_MPLAYER_ENDSTATS, 100, 480, false);
		break;

		/*case MENU_MPLAYER_CHATROOM:
			DrawChatMenu();
		break;*/
	}
}

void CGameMenu::ResetView()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);	// Clear Screen And Depth Buffer
	glLoadIdentity();									// Reset The Current Modelview Matrix

	gluLookAt(0.0, 1.0, 0.0, 0.0, 0.0, -3.0, 0.0, 1.0, 0.0);	// Ein wenig von oben auf das Modell schauen
}

void CGameMenu::DrawDefaultMenu(char* szTitle, int nMenuID, int nMenuTitlesOffsetX, int nMenuTitlesOffsetY, bool bDrawModel)
{
	if (bDrawModel)
	{
		ResetView();
		m_pPlayerModel->UpdateForShowCase();	//Drehanimation des Modells
		m_pPlayerModel->Draw();
	}

	C2DTools::Enter2DMode();
		if (!bDrawModel)
		{
			MenuFadeIn();
		}

		g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.8, 0.0);

		DrawMenuHeader(szTitle);

		glColor3ub(0, 150, 0);
		C2DTools::DrawLine(100, SCREEN_RES_Y - 200, SCREEN_RES_X - 100, SCREEN_RES_Y - 200);

		DrawMenuTitles(nMenuID, nMenuTitlesOffsetX, nMenuTitlesOffsetY);
	C2DTools::Leave2DMode();
}


void CGameMenu::DrawGameRoom()
{
	ResetView();
	m_pPlayerModel->UpdateForShowCase();	//Drehanimation des Modells
	m_pPlayerModel->Draw();

	SGameInfo gameInfo = m_vAvailableGames[m_nSelectedGame];
	char szTitle[256];
	sprintf(szTitle, "TRAXI 2.0 | %s", gameInfo.szName);

	C2DTools::Enter2DMode();
		g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.8, 0.0);

		DrawMenuHeader(szTitle);

		glColor3ub(0, 150, 0);
		C2DTools::DrawLine(100, SCREEN_RES_Y - 200, SCREEN_RES_X - 100, SCREEN_RES_Y - 200);

		DrawMenuTitles(MENUID_MPLAYER_GAMEROOM, 100, 540, true);
	C2DTools::Leave2DMode();
}


void CGameMenu::DrawChatMenu()
{
	ResetView();
	m_pPlayerModel->UpdateForShowCase();	//Drehanimation des Modells
	m_pPlayerModel->Draw();

	SGameInfo gameInfo = m_vAvailableGames[m_nSelectedGame];
	char szTitle[256];
	sprintf(szTitle, "TRAXI 2.0 | %s", gameInfo.szName);

	C2DTools::Enter2DMode();
		g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.8, 0.0);

		DrawMenuHeader(szTitle);

		glColor3ub(0, 150, 0);
		C2DTools::DrawLine(100, SCREEN_RES_Y - 200, SCREEN_RES_X - 100, SCREEN_RES_Y - 200);

		//DrawMenuTitles(nMenuID, nMenuTitlesOffsetX, nMenuTitlesOffsetY);
	C2DTools::Leave2DMode();
}


void CGameMenu::MenuFadeIn()
{
	if (m_nFadeInFrames <= 15)
	{
		//glPushAttrib(GL_COLOR_BUFFER_BIT);
		glEnable(GL_BLEND);
		glColor3ub(3, 3, 3);
		C2DTools::DrawRect(80, 100, SCREEN_RES_X - 160, SCREEN_RES_Y - 200);	/// Immer sch�n eine Schicht nach der anderen dr�berzeichnen
		glDisable(GL_BLEND);
		//glPopAttrib();

		glColor3ub(0, 200, 0);
		C2DTools::DrawRect(80, 100, SCREEN_RES_X - 160, SCREEN_RES_Y - 200, false);

		m_nFadeInFrames++;
	}
}

void CGameMenu::DrawMenuHeader(char* szHeaderTitle)
{
	g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.8, 0.0);

	g_pText->DrawStr(100, SCREEN_RES_Y - 190, szHeaderTitle);

	glColor3ub(0, 150, 0);
	C2DTools::DrawLine(100, SCREEN_RES_Y - 200, SCREEN_RES_X - 100, SCREEN_RES_Y - 200);
	glColor3ub(255, 255, 255);
}

void CGameMenu::DrawMenuTitles(int nMenuTitlesID, int nOffsetX, int nOffsetY, bool bHorizontal)
{
	char szInfo[256];
	vector< SPlayer > vPlayers;

	switch (nMenuTitlesID)
	{
		default:
		case MENUID_OK:
			m_vMenuTitles.push_back("OK");
		break;

		case MENUID_BACK:
			m_vMenuTitles.push_back("Zurueck");
		break;

		case MENUID_MAIN:
			m_vMenuTitles.push_back("Start");
			m_vMenuTitles.push_back("Level laden");
			m_vMenuTitles.push_back("Multiplayer");
			m_vMenuTitles.push_back("Hilfe");
			m_vMenuTitles.push_back("Credits");
			m_vMenuTitles.push_back("Programm beenden");
		break;

		case MENUID_GAME_PAUSED:
			m_vMenuTitles.push_back("Weiter");
			//m_vMenuTitles.push_back("Hilfe");
			m_vMenuTitles.push_back("Zum Hauptmenu");
			m_vMenuTitles.push_back("Programm beenden");
		break;

		case MENUID_MPLAYER:
			m_vMenuTitles.push_back("Spieler Einstellungen");
			m_vMenuTitles.push_back("LAN-Spiel erstellen");
			m_vMenuTitles.push_back("Internet Spiel erstellen");
			m_vMenuTitles.push_back("Spiel beitreten");
			m_vMenuTitles.push_back("Zurueck");
		break;

		case MENUID_MPLAYER_PLAYERSETUP:
			m_vMenuTitles.push_back("Spielername:");
			m_vMenuTitles.push_back("Zurueck");

			//Eingabemaske zeichnen
			glColor3f(1.0, 1.0, 1.0);
			C2DTools::DrawRect(310, 347, 200, 20, false);

			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.8, 0.0);
			g_pText->DrawStr(313, 352, m_vsInput[0].c_str());
		break;

		case MENUID_MPLAYER_CREATE_INET:
			g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);
			g_pText->DrawStr(100, SCREEN_RES_Y - 220, "Bitte die IP des TRAXI-Servers eingeben.");

		case MENUID_MPLAYER_JOIN:
			m_vMenuTitles.push_back("IP-Adresse:");

			if (nMenuTitlesID == MENUID_MPLAYER_JOIN)
			{
				if (m_bIPEntered)
					m_vAvailableGames = m_pGame->GetClient()->GetGames();

				m_pGame->GetClient()->LockMutex();

				if (m_bIPEntered
					&& SDL_GetTicks() < m_fConnectionStarted + CONNECTION_TIMEOUT
					&& (m_vAvailableGames.size() < 1 || nMenuTitlesID == MENUID_MPLAYER_CREATE_INET))
				{
					g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.9, 0.0);
					g_pText->DrawStr(520, 352, "Verbinde...");
				}

				SGameInfo gameInfo;
				for (int i = 0; i < m_vAvailableGames.size(); i++)
				{
					gameInfo = m_vAvailableGames[i];
					sprintf(szInfo, "%s%c (%d/%d)",
						gameInfo.szName,
						((gameInfo.bUsePassword) ? '*' : ' '),
						(int)gameInfo.nNumPlayers,
						(int)gameInfo.nMaxPlayers);

					m_vMenuTitles.push_back(string(szInfo));

					if (m_bJoinError)
					{
						g_pText->DrawStr(520, SCREEN_RES_Y - nOffsetY - 30 * (i + 1), "Fehler beim Beitreten!");
					}
				}

				m_pGame->GetClient()->UnlockMutex();
			}

			//Eingabemaske fuer IP zeichnen
			glColor3f(1.0, 1.0, 1.0);
			C2DTools::DrawRect(310, 347, 200, 20, false);

			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.8, 0.0);
			g_pText->DrawStr(313, 352, m_vsInput[0].c_str());

			if (nMenuTitlesID == MENUID_MPLAYER_CREATE_INET && m_bIPEntered)
			{
				//Spiel erstellen Button fuer den Dedi
				m_vMenuTitles.push_back("Spiel erstellen");
			}
			else
			{
				//Fehlermeldungen beim Spiel beitreten
				g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.9, 0.0, 0.0);
				if (m_bConnectionError)
				{
					g_pText->DrawStr(520, 352, "Fehler beim Verbinden!");
				}
				else if (m_bIPEntered
						&& SDL_GetTicks() > m_fConnectionStarted + CONNECTION_TIMEOUT
						&& m_vAvailableGames.size() < 1)
				{
					g_pText->DrawStr(520, 352, "Keine Spiele auf dem Server!");
				}
			}

			m_vMenuTitles.push_back("Zurueck");
		break;

		case MENUID_MPLAYER_JOIN_PASSWD:
			m_vMenuTitles.push_back("Passwort:");

			//Eingabemaske fuer Passwort zeichnen
			glColor3f(1.0, 1.0, 1.0);
			C2DTools::DrawRect(310, 347, 200, 20, false);

			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.8, 0.0);
			g_pText->DrawStr(313, 352, m_vsInput[0].c_str());

			m_vMenuTitles.push_back("Weiter");
			m_vMenuTitles.push_back("Zurueck");
		break;

		case MENUID_MPLAYER_GAMEROOM:
		{
			vPlayers = m_pGame->GetClient()->GetPlayer();
			SGameInfo* pGameInfo = m_pGame->GetClient()->GetRunningGame();

			if (pGameInfo->eState == SERVERGAME_RUNNING)
			{
				ActionStartMPGame();
				return;
			}

			bool bPlayerReady = false;

			g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.8, 0.0);
			char szGamePlayerInfo[256];
			sprintf(szGamePlayerInfo, "Teilnehmer: %d/%d", vPlayers.size(), m_vAvailableGames[m_nSelectedGame].nMaxPlayers);
			g_pText->DrawStr(100, SCREEN_RES_Y - 220, szGamePlayerInfo);
			glColor3f(0.0f, 0.8f, 0.0f);
			C2DTools::DrawRect(100, SCREEN_RES_Y - 490, 150, 260, false);

			m_pGame->GetClient()->LockMutex();
			for (int i = 0; i < vPlayers.size(); i++)
			{
				g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);
				g_pText->DrawStr(120, SCREEN_RES_Y - 250 - 30 * i, vPlayers[i].sInfo.szName);

				if (vPlayers[i].sIngameInfo.eState == PLAYER_READY)
				{
					glColor3f(0.0f, 0.9f, 0.0f);
				}
				else
				{
					glColor3f(0.9f, 0.0f, 0.0f);
				}

				C2DTools::DrawRect(108, SCREEN_RES_Y - 247 - 30 * i, 4, 4);

				if (vPlayers[i].sInfo.nID == m_pGame->GetClient()->GetLocalPlayerID()
					&& vPlayers[i].sIngameInfo.eState == PLAYER_READY)
				{
					bPlayerReady = true;
				}
			}
			m_pGame->GetClient()->UnlockMutex();

			m_vMenuTitles.push_back("Zurueck");

			if (bPlayerReady)
			{
				m_vMenuTitles.push_back("Warte...");
			}
			else
			{
				m_vMenuTitles.push_back("Bereit!");
			}
		}
		break;

		case MENUID_LEVELS:
			//Levelauswal. Hier ist etwas mehr Arbeit noetig...
			//Zuerst Infotext anzeigen
			g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);
			g_pText->DrawStr(100, SCREEN_RES_Y - 230, "Hier werden alle bisher freigeschaltenen Levels angezeigt.");
			g_pText->DrawStr(100, SCREEN_RES_Y - 250, "In Klammern steht die bisher maximal erreichte Punktzahl.");

		case MENUID_MPLAYER_CREATE:
		{
			//Jetzt Variablen festlegen
			vector<string> vLevels;
			int nPoints = 0;
			map<string, int> mapHighscore = m_pSaveGame->GetHighscore();
			map<string, int>::iterator itFind;

			if (nMenuTitlesID == MENUID_MPLAYER_CREATE)
			{
				g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);
				g_pText->DrawStr(200, SCREEN_RES_Y - 240, "Verfuegbare Levels:");
				g_pText->DrawStr(400, SCREEN_RES_Y - 240, "Ausgewaehlte Levels:");
			}

			//Alle Leveldateien einlesen
			if (nMenuTitlesID == MENUID_LEVELS)
			{
#ifdef WIN32
				CFileTools::GetFiles("Data/Levels/taxi/taxi*.txt", vLevels);	//Windows brauch umbedingt Wildcards...
#else
				CFileTools::GetFiles("Data/Levels/taxi/", vLevels);
#endif
			}
			else
			{
#ifdef WIN32
				CFileTools::GetFiles("Data/Levels/race/*.txt", vLevels);	//Windows brauch umbedingt Wildcards...
#else
				CFileTools::GetFiles("Data/Levels/race/", vLevels);
#endif
			}

			char szTmp[256];
			char chTmp;

			//Menuepunkt-Liste mit freigeschaltenen Levels fuellen
			int nLevelNum = 1;
			for (int i = 0; i < (int) vLevels.size(); i++)
			{
				if (vLevels[i].substr(0, 4) == "taxi"
					&& atoi(vLevels[i].substr(4, vLevels[i].find_first_of('.')).c_str()) > m_pSaveGame->GetSaveDataPointer()->nUnlockedLevel)
				{
					continue;
				}

				//Highscore fuer dieses Level herausfinden
				if (nMenuTitlesID == MENUID_LEVELS)
				{
					sprintf(szTmp, "Data/Levels/taxi/%s", vLevels[i].c_str());
					itFind = mapHighscore.find(string(szTmp));
					if (itFind != mapHighscore.end())
						nPoints = itFind->second;
					else
						nPoints = 0;

					sprintf(szTmp, "Level %d (%d Punkte)", nLevelNum, nPoints);
				}
				else
				{
					strcpy(szTmp, vLevels[i].c_str());
				}

				m_vMenuTitles.push_back(string(szTmp));

				//Ausgewaehlte Levels fuer Custom Server anzeigen
				if (nMenuTitlesID == MENUID_MPLAYER_CREATE)
				{
					for (int j = 0; j < m_vSelectedMaps.size(); j++)
					{
						if (m_vSelectedMaps[j] == i)
						{
							g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.0, 0.8, 0.0);
							g_pText->DrawStr(400, SCREEN_RES_Y - 290 - j * 20, vLevels[i].c_str());
						}
					}
				}

				nLevelNum++;
			}

			if (nMenuTitlesID == MENUID_MPLAYER_CREATE)
			{
				m_vMenuTitles.push_back("Weiter");
			}

			m_vMenuTitles.push_back("Zurueck");
		}
		break;

		case MENUID_MPLAYER_CREATECONFIG:
		{
			if (m_bConnectionError)
			{
				g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.9, 0.0, 0.0);
				g_pText->DrawStr(520, 352, "Fehler beim Beitreten!");
			}

			//Eingabemaske fuer Spielname zeichnen
			m_vMenuTitles.push_back("Spielname:");
			glColor3f(1.0, 1.0, 1.0);
			C2DTools::DrawRect(310, 347, 200, 20, false);

			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.8, 0.0);
			g_pText->DrawStr(313, 352, m_vsInput[0].c_str());

			//Eingabemaske fuer Passwort zeichnen
			m_vMenuTitles.push_back("Max. Spieler:");
			glColor3f(1.0, 1.0, 1.0);
			C2DTools::DrawRect(310, 317, 30, 20, false);

			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.8, 0.0);
			g_pText->DrawStr(313, 322, m_vsInput[1].c_str());

			//Eingabemaske fuer Spieleranzahl zeichnen
			m_vMenuTitles.push_back("Passwort:");
			glColor3f(1.0, 1.0, 1.0);
			C2DTools::DrawRect(310, 287, 200, 20, false);

			g_pText->SetFont(GLUT_BITMAP_9_BY_15, 0.0, 0.8, 0.0);
			g_pText->DrawStr(313, 292, m_vsInput[2].c_str());

			m_vMenuTitles.push_back("Weiter");
			m_vMenuTitles.push_back("Zurueck");
		}
		break;

		case MENUID_MPLAYER_ENDSTATS:
		{
			g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);

			vector<SPlayer> vPlayerStats = m_pGame->GetGui()->GetPlayerStats();
			for (int i = 0; i < vPlayerStats.size(); i++)
			{
				if (vPlayerStats[i].sInfo.nID == m_pGame->GetClient()->GetLocalPlayerID())
				{
					g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);

					if (i == 0)
					{
						g_pText->DrawStr(100, SCREEN_RES_Y - 230, "Du bist der Sieger!");
					}
					else
					{
						g_pText->DrawStr(100, SCREEN_RES_Y - 230, "Schade, schade, zum Sieg hat's nicht gereicht!");
					}
				}
				else
				{
					g_pText->SetFont(GLUT_BITMAP_8_BY_13, 0.8, 0.8, 0.8);
				}

				g_pText->DrawStr(100, SCREEN_RES_Y - (260 + 20 * i),
					"#%d: %s mit %d Siegen",
					i + 1, vPlayerStats[i].sInfo.szName, vPlayerStats[i].sIngameInfo.nRoundsWon
				);
			}

			m_vMenuTitles.push_back("OK");
		}
		break;
	}

	//Anzeige von maximal 6 Men�punkten gleichzeitig. Wenn mehr, wird nach unten gescrollt!
	int nBegin = m_nSelectedMenuTitle - 5;
	int nEnd = m_nSelectedMenuTitle + 1;
	int nScrollOffset;

	if (nBegin < 0)
		nBegin = 0;

	if (nEnd < 6)
		nEnd = 6;

	if (nEnd > (int) m_vMenuTitles.size())
		nEnd = m_vMenuTitles.size();

	int nHorizontalSpace = 600 / m_vMenuTitles.size();

	//Men�punkte darstellen
	for (int i = nBegin; i < nEnd; i++)
	{
		nScrollOffset = i - nBegin;

		if (i == m_nSelectedMenuTitle)
			g_pText->SetFont(GLUT_BITMAP_HELVETICA_18, 0.0 * m_fSelectedColorIntensity, 1.0 * m_fSelectedColorIntensity, 0.0 * m_fSelectedColorIntensity);
		else
			g_pText->SetFont(GLUT_BITMAP_HELVETICA_18, 0.0, 0.8, 0.0);

		if (bHorizontal)
			g_pText->DrawStr(nOffsetX + nHorizontalSpace * nScrollOffset, SCREEN_RES_Y - nOffsetY, m_vMenuTitles.at(i).c_str());
		else
			g_pText->DrawStr(nOffsetX, SCREEN_RES_Y - nOffsetY - 30 * nScrollOffset, m_vMenuTitles.at(i).c_str());
	}
}

void CGameMenu::DrawMessageBox(char* szTitle, int nTextID, int nMenuTitlesID, bool bDrawModel, bool bFadeIn)
{
	if (bDrawModel)
	{
		m_pPlayerModel->UpdateForShowCase();
		m_pPlayerModel->Draw();
	}

	C2DTools::Enter2DMode();
		if (bFadeIn)
			MenuFadeIn();

		DrawMenuHeader(szTitle);

		g_pText->SetFont(GLUT_BITMAP_8_BY_13, 1.0, 1.0, 1.0);

		switch (nTextID)
		{
			case TEXTID_LEVEL_END:
				g_pText->DrawStr(100, SCREEN_RES_Y - 250, "Gratulation! Du hast dich erfolgreich durch den Datenjungle");
				g_pText->DrawStr(100, SCREEN_RES_Y - 270, "gekaempft! Ein CRC-Check ergab, dass deine binaere Fracht ohne");
				g_pText->DrawStr(100, SCREEN_RES_Y - 290, "groessere Schaeden angekommen ist.");
				g_pText->DrawStr(100, SCREEN_RES_Y - 330, "Eingesammelte Punkte: %d", m_pGame->GetPlayerObj()->nScore);
				g_pText->DrawStr(100, SCREEN_RES_Y - 350, "Bonus-Punkte: %d", m_pGame->GetPlayerObj()->nBonusScore);
				g_pText->DrawStr(100, SCREEN_RES_Y - 370, "Level %d: Abgeschlossen mit %d Punkten.", m_pGame->GetCurrentLevel(), m_pGame->GetPlayerObj()->nScore + m_pGame->GetPlayerObj()->nBonusScore);
			break;

			case TEXTID_GAME_COMPLETED:
				g_pText->DrawStr(100, SCREEN_RES_Y - 250, "Gratulation! Du hast das %d. und damit letzte Level erreicht! Du hast", GAME_MAX_LEVEL);
				g_pText->DrawStr(100, SCREEN_RES_Y - 270, "deinen grossen Auftrag erfuellt, alle Daten sind schliesslich in der CPU");
				g_pText->DrawStr(100, SCREEN_RES_Y - 290, "angelangt Aber das geht noch schneller, wuerde ich mal meinen... Zack zack,");
				g_pText->DrawStr(100, SCREEN_RES_Y - 310, "die Bytes wollen schneller transportiert werden! Knacke die Highscore!");
			break;

			case TEXTID_CREDITS:
				g_pText->DrawStr(100, SCREEN_RES_Y - 250, "Belegarbeit CG / VSYS - SoSe 08 FHTW Berlin");
				g_pText->DrawStr(100, SCREEN_RES_Y - 270, "Traxi - Ein Multiplayer Tron / Space Taxi Konglomerat in C++ / OpenGL / Sockets ");
				g_pText->DrawStr(100, SCREEN_RES_Y - 290, "PROGRAMMIERUNG:");
				g_pText->DrawStr(120, SCREEN_RES_Y - 310, "Michael Witt");
				g_pText->DrawStr(120, SCREEN_RES_Y - 330, "Peter Schmidt");
				g_pText->DrawStr(120, SCREEN_RES_Y - 350, "Andre Stueker");
				g_pText->DrawStr(120, SCREEN_RES_Y - 370, "Sebastian Mueller");
				g_pText->DrawStr(120, SCREEN_RES_Y - 390, "Markus Konrad");
				g_pText->DrawStr(100, SCREEN_RES_Y - 410, "MODELLING:");
				g_pText->DrawStr(120, SCREEN_RES_Y - 430, "Peter Schmidt");
			break;

			case TEXTID_HELP:
				g_pText->DrawStr(100, SCREEN_RES_Y - 250, "TASTENBELEGUNG:");
				g_pText->DrawStr(120, SCREEN_RES_Y - 270, "F1..............Diese Hilfe");
				g_pText->DrawStr(120, SCREEN_RES_Y - 290, "Pfeiltasten.....Traxi-Steuerung");
				g_pText->DrawStr(120, SCREEN_RES_Y - 310, "Leertaste.......Schub nach oben");
				g_pText->DrawStr(120, SCREEN_RES_Y - 330, "STRG links......Schub nach unten");
				g_pText->DrawStr(120, SCREEN_RES_Y - 350, "ESC.............Menue");
				g_pText->DrawStr(120, SCREEN_RES_Y - 370, "W...............Kamerawinkel +");
				g_pText->DrawStr(120, SCREEN_RES_Y - 390, "S...............Kamerawinkel -");
				g_pText->DrawStr(120, SCREEN_RES_Y - 410, "E...............Kamerazoom +");
				g_pText->DrawStr(120, SCREEN_RES_Y - 430, "D...............Kamerazoom -");
			break;

			case TEXTID_GAME_OVER:
				g_pText->DrawStr(100, SCREEN_RES_Y - 250, "Tja, Murphys Gesetz hat mal wieder gesiegt...");
				g_pText->DrawStr(100, SCREEN_RES_Y - 290, "Aber nicht verzagen, es kann nur noch besser werden!");
			break;
		}

		DrawMenuTitles(nMenuTitlesID, 100, 480, true);
	C2DTools::Leave2DMode();
}
