#ifndef GAMEMENU_HPP
#define GAMEMENU_HPP

#include <math.h>
#include <string>
#include <sstream>
#include <vector>
#include <cctype>

#include "../client/PlatformFramework.hpp"
#include "../globals.hpp"
#include "../gfx/Text.hpp"
#include "../gfx/Gui.hpp"
#include "../tools/2DTools.hpp"
#include "../tools/Timer.hpp"
#include "../tools/Singleton.hpp"
#include "../tools/FileTools.hpp"
#include "../net/NetworkTypes.hpp"

#include "Player.hpp"
#include "Game.hpp"
#include "SaveGame.hpp"

#define CONNECTION_TIMEOUT 5000

//Text IDs
#define TEXTID_LEVEL_END 1
#define TEXTID_CREDITS 2
#define TEXTID_GAME_COMPLETED 3
#define TEXTID_GAME_OVER 4
#define TEXTID_HELP 5

//Menu IDs
#define MENUID_OK 0
#define MENUID_MAIN 1
#define MENUID_GAME_PAUSED 2
#define MENUID_BACK 3
#define MENUID_LEVELS 4
#define MENUID_MPLAYER 5
#define MENUID_MPLAYER_CREATE 6
#define MENUID_MPLAYER_PLAYERSETUP 7
#define MENUID_MPLAYER_JOIN 8
#define MENUID_MPLAYER_CHATROOM 9
#define MENUID_MPLAYER_GAMEROOM 10
#define MENUID_MPLAYER_CREATECONFIG 11
#define MENUID_MPLAYER_JOIN_PASSWD 12
#define MENUID_MPLAYER_ENDSTATS 13
#define MENUID_MPLAYER_CREATE_INET 14

class CGame;
class CPlayer;

#define g_pGameMenu CGameMenu::Get()

/// Die Spielmenu-Klasse (als Singleton). Sie sorgt fuer die Anzeige aller 2D-Menueelemente (aber nicht des GUIs!)
/// wie Hauptmenue, Pausemenue usw.
/// Die Anzeige des jeweiligen Menues ist abhaengig vom aktuellen Programm- und Spielstatus.
class CGameMenu : public TSingleton<CGameMenu>
{
	public:
		/// Konstruktor. Legt Standardwerte fest
		CGameMenu();

		/// Dekonstruktor. Gibt Speicher frei
		~CGameMenu();

		/// Funktion, die zur Initialisierung des GameMenus die Zeiger auf Programm- / Spielstatus, Spielinstanz
		/// und SaveGame Instanz setzt
		/// @param peProgStatus Zeiger auf Programmstatus
		/// @param peGameStatus Zeiger auf Spielestatus
		/// @param pGame Spielinstanz
		/// @param pSaveGame SaveGame Instanz
		void Init(EProgramStatus* peProgStatus, EGameStatus* peGameStatus, CGame* pGame, CSaveGame* pSaveGame);

		/// Zeigt das aktuelle Menue an, je nach Programm und Spielstatus
		/// Wird in der run() Funktion von CGame aufgerufen
		void Draw();

		/// Aktualisiert Menuedaten, ruft ProcessEvents() auf
		void Update();

	private:
		/// Standardmenu anzeigen
		/// @param szTitle Menuetitel
		/// @param nMenuID ID des Menues, das angezeigt werden soll (MENUID_* Definition)
		///	@param nMenuTitlesOffsetX X-Koordinate fuer die Menueposition
		///	@param nMenuTitlesOffsetY Y-Koordinate fuer die Menueposition
		/// @param bDrawModel Das sich rotierende TRAXI-Model im Header anzeigen?
		void DrawDefaultMenu(char* szTitle, int nMenuID, int nMenuTitlesOffsetX = 200,
			int nMenuTitlesOffsetY = 250, bool bDrawModel = true);

		/// Anzeige Hilfemenue mit Tastenbelegung
		void DrawHelpMenu();

		/// Chatroom anzeigen
		void DrawChatMenu();

		/// Spielerraum anzeigen mit Auflistung aller Spieler
		void DrawGameRoom();

		/// Allgemeine Messagebox-Anzeige
		/// @param szTitle Titel der Messagebox
		/// @param nTextID ID des Textes, der angezeigt werden soll (TEXTID_* Definition)
		/// @param nMenuID ID des Menues, das angezeigt werden soll (MENUID_* Definition)
		/// @param bDrawModel Das sich rotierende TRAXI-Model im Header anzeigen?
		/// @param bFadeIn Menue-FadeIn?
		void DrawMessageBox(char* szTitle, int nTextID, int nMenuTitlesID,
			bool bDrawModel = false, bool bFadeIn = true);

		/// Menue ueberschrift anzeigen
		/// @param szHeaderTitle Ueberschrift des Menues
		void DrawMenuHeader(char* szHeaderTitle);

		/// Auswaehlbare Menuepunkte anzeigen
		/// @param nMenuID ID des Menues, das angezeigt werden soll (MENUID_* Definition)
		///	@param nOffsetX X-Koordinate fuer die Menueposition
		///	@param nOffsetY Y-Koordinate fuer die Menueposition
		/// @param bHorizontal Menupunkte horizontal anzeigen oder vertikal?
		void DrawMenuTitles(int nMenuTitlesID, int nOffsetX, int nOffsetY, bool bHorizontal = false);

		/// Fade-In fuer das Menue
		void MenuFadeIn();

		/// Abfrage von Standard-Events, die fuer jedes Menue gueltig sind
		void ProcessEvents();

		/// Abfrage fuer Menue-Spezifische Events
		void ProcessMenuSelect();

		/// Einem Spiel beitreten
		/// @param szPasswd Passwort fuer das Spiel
		void ActionJoin(const char* szPasswd);

		/// MP-Spiel starten
		void ActionStartMPGame();

		/// Tastatur-Eingabe aktivieren
		/// @param pfInputHandler Funktion, die NACH der Eingabe des Textes aufgerufen wird
		/// @param nMaxChars Maximale Eingabelaenge
		/// @param nTypingField ID des Eingabefelds
		void StartTyping(void (CGameMenu::*pfInputHandler)(void), int nMaxChars, int nTypingField);

		/// Tastatur-Eingabe abfangen und darstellen
		void ProcessTyping();

		/// Wird nach der Eingabe des Spielernamens aufgerufen
		void HandlePlayerNameInput();

		/// Wird nach der Eingabe der IP aufgerufen
		void HandleIPInput();

		/// Ansicht fuer das Menue zuruecksetzen
		void ResetView();

		/// In neuen Programm/Spielmodus wechseln
		/// @param eProgStatus Neuer Programmstatus
		/// @param eGameStatus Neuer Spielstatus
		void EnterNewMode(EProgramStatus eProgStatus, EGameStatus eGameStatus);



		bool m_bIPEntered;			///< Wurde die IP eingetippt?
		int m_nRegisterTyping;		///< Tastatureingaben für Namenseingabe ueberwachen. m_nRegisterTyping ist dabei die ID des Eingabefelds. -1 bedeutet keine Eingabe
		int m_nTypingTextMaxChars;	///< Max. Zeichen für Tastatureingabe
		vector< string > m_vsInput;	///< Eingegebene Zeichenfolge pro Eingabefeld
		string m_sEnteredIP;		///< Eingegebene IP
		void (CGameMenu::*m_pfInputHandler)(void);	///< Zeiger auf Funktion, die NACH Eingabe des Textes aufgerufen werden soll

		bool m_bConnectionError;		///< Ist ein Fehler beim Verbinden zum Server aufgetreten?
		bool m_bJoinError;				///< Ist ein Fehler beim Beitreten zum Spiel aufgetreten?
		float m_fConnectionStarted;		///< Speichert die Zeit, wann begonnen wurde zu Verbinden wegen Connection-Timeout
		EProgramStatus* m_peProgStatus;	///< Aktueller Programmstatus
		EGameStatus* m_peGameStatus;	///< Aktueller Spielstatus
		CGame* m_pGame;					///< Spielinstanz
		CServer* m_pLANServerThread;	///< Zeiger auf den LANServer fuer selbst erstellten Server
		CSaveGame* m_pSaveGame;			///< Savegame-Instanz
		int m_nSelectedMenuTitle;		///< Ausgewaehlter Menuepunkt
		vector<string> m_vMenuTitles;	///< Vektor von Menuepunten
		int m_nFadeInFrames;			///< Counter fuer FadeIn-Effekt
		float m_fSelectedColorIntensity;///< Aktuelle Helligkeit fuer Menuehervohebung
		CPlayer* m_pPlayerModel;		///< Taximodell zum anzeigen und angeben
		SPlayerObj* m_pPlayerObj;		///< Zeiger auf Fake-Spielerdaten die nur zum Anzeigen des Modells im Hauptmenue gebraucht werden

		vector< SGameInfo > m_vAvailableGames;	///< Vektor von Spielen, die auf dem Server existieren
		vector< u8 > m_vSelectedMaps;			///< Ausgewaehlte Maps fuer Custom Server
		int m_nSelectedGame;					///< Ausgewaehltes Spiel
};

#endif
