#ifndef PLAYERCTRL_HPP
#define PLAYERCTRL_HPP

#include <map>
#include "../globals.hpp"
#include "../tools/Timer.hpp"

#define DEATH_ANIM_SECONDS 3.0
#define CRASH_ANIM_SECONDS 3.0

class CPlayer;
class CGui;
class CWorld;

/// Player Controller Klasse zur Verwaltung mehrerer Spieler
/// Basisklasse fuer Single-Playerfunktionalitt
class CPlayerCtrl
{
	public:
		/// Konstruktor. Setzt die Standardwerte fuer das SP-Spiel
		/// @param peProgStatus Programmstatus
		/// @param peGameStatus Spielstatus
		CPlayerCtrl(EProgramStatus* peProgStatus, EGameStatus* peGameStatus);

		/// Dekonstruktor. Raeumt auf.
		virtual ~CPlayerCtrl(void);

		/// Den Zeiger auf die GUI-Instanz setzen
		/// @param pGui Zeiger auf die GUI-Instanz
		void SetGuiPointer(CGui* pGui) { m_pGui = pGui; }

		/// Den Zeiger auf die Level-Instanz setzen
		/// @param pWorld Zeiger auf die Level-Instanz
		virtual void SetWorldPointer(CWorld* pWorld);

		/// Player-Instanz fuer einen Spieler zurueck geben
		/// @param nID Spieler mit dieser ID. Beim Standardwert -1 gib den lokalen Spieler zurueck
		/// @return Player-Instanz fuer einen Spieler mit dieser ID
		CPlayer* GetPlayer(int nID = -1);

		/// Spieler-Daten fuer einen Spieler zurueck geben
		/// @param nID Spieler mit dieser ID. Beim Standardwert -1 gib den lokalen Spieler zurueck
		/// @return Spieler-Instanz fuer einen Spieler mit dieser ID
		SPlayerObj* GetPlayerData(int nID = -1);

		/// Bonus-Score berechnen nach Anzahl der eingesammelten Sterne
		/// Single Player only!
		void CalcPlayerBonusScore();

		/// Zurueck setzen des lokalen Spielers auf Standardwerte
		virtual void Reset();

		/// Update fuer den lokalen Spieler durchfuehren
		virtual void Update();

		/// Den lokalen Spieler zeichnen
		/// @param bDrawAxis Zu Debugzwecken die Achsen dazu zeichnen
		virtual void Draw(bool bDrawAxis = true);

	protected:
		/// Zuruecksetzen der Werte nach dem Tod fuer einen Spieler
		/// @param nDeathBlockX Block des Todes - X Koord.
		/// @param nDeathBlockY Block des Todes - Y Koord.
		/// @param nDeathBlockZ Block des Todes - Z Koord.
		/// @param nID Fuer einen Spieler mit dieser ID, standard den lokalen Spieler
		virtual void ResetAfterDeath(const int nDeathBlockX, const int nDeathBlockY, const int nDeathBlockZ, int nID = -1);

		/// Spieler dieser ID zeichnen
		/// @param i Spieler-ID
		/// @param bDrawAxis Zu Debugzwecken die Achsen dazu zeichnen
		/// @param bDrawName Den Namen darunter zeichnen
		void DrawPlayer(int i = -1, bool bDrawAxis = false, bool bDrawName = false);

		/// Absturzanimation zeichnen fuer diese ID
		/// @param i Spieler-ID
		void DrawCrash(int i = -1);

		/// Todesanimation zeichnen fuer diese ID
		/// @param i Spieler-ID
		void DrawDeath(int i = -1);

		SPlayerObj* m_pLocPlayerData;			///< Zeiger auf lokale Spielerdaten
		CPlayer* m_pPlayerInstance;				///< Zeiger auf lokalen Spieler
		map < int, SPlayerObj >*  m_pmPlayers;	///< Zeiger auf Map von angemeldeten Spielern (aus CClient) oder ein selbst erstellter lokaler Spieler (Singleplayer)
		CGui* m_pGui;							///< Zeiger auf GUI
		CWorld* m_pWorld;						///< Zeiger aufs Level
		EProgramStatus* m_peProgStatus;			///< Zeiger auf Programmstatus
		EGameStatus* m_peGameStatus;			///< Zeiger auf Spielstatus
		int m_nLocalPlayerID;					///< ID des lokalen Spielers
};

#endif
