#ifndef WORLD_HPP
#define WORLD_HPP

#include <cmath>
#include <fstream>
#include <string>
#include <algorithm>
#include <vector>
#include <map>
#include <sstream>

#include "../client/PlatformFramework.hpp"

#include "../globals.hpp"
#include "../client/Framework.hpp"
#include "../gfx/Texture.hpp"
#include "../tools/Timer.hpp"
#include "../tools/Vec3D.hpp"
#include "../tools/FileTools.hpp"

#include "Player.hpp"

using namespace std;

/// Makros als Shortcut
#define CHECK_SIDE(off1, off2, side1, side2) ( (nCollPlayerID = PlayerSideCollides(off1, off2, side1, side2)) > -1 )

/// Definition von speziellen Blocktypen
#define SPECIAL_BLOCK_NONE 0
#define SPECIAL_BLOCK_START 1
#define SPECIAL_BLOCK_END 2
#define SPECIAL_BLOCK_FUEL 3
#define SPECIAL_BLOCK_BONUS 4

/// Globale Blockskalierung:
#define BLOCK_SCALE 1.0f

/// Blockseiten:
#define BLOCK_NOSIDE 0
#define BLOCK_TOP 1
#define BLOCK_RIGHT 2
#define BLOCK_BOTTOM 3
#define BLOCK_LEFT 4

/// Blockstruktur
struct SBlock
{
	int nBeginHeight;	///< Beginn-H�he des Blocks
	int nEndHeight;		///< End-H�he des Blocks
	int nSpecial;		///< Spezial-ID wie Tankfeld, Startfeld, etc.
	int nTexture;		///< Textur-ID siehe CTexture
};

struct SBlockCoord
{
	int x;
	int y;
	int z;
};

#define CAM_MIN_DISTANCE 1.0
#define CAM_MAX_DISTANCE 10.0
#define CAM_MIN_ANGLE 1.0
#define CAM_MAX_ANGLE 89.0
#define CAM_HEIGHT 3.0

/// Kollisionsseiten
enum ECollision {NONE, OUT_OF_FIELD, ABOVE, BENEATH, LEFT, RIGHT, TOP, BOTTOM};

/// Die Welt. Ein 3D-Level aus Bloecken verschiedener Groesse und verschiedener
/// Art. Grob gesagt.
/// Die Klasse stellt Funktionen zum Laden des Levels aus einer Textdatei
/// und zum zeichnen dieses Levels zur Verfuegung. Ausserdem wird die Bewegung
/// und das Umherblicken in dieser Welt realisiert.
class CWorld
{
	public:
		/// Konstruktor. Setzt Standardwerte wie Startposition
		CWorld();

		/// Dekonstruktor. Gibt den Speicher frei.
		~CWorld();

		/// Zeiger für Spielerobjekt setzen
		void SetPlayer(SPlayerObj* pPlayerObj) { m_pPlayerObj = pPlayerObj; }

		/// Zeiger auf Spielervektor setzen (Multiplayer only!)
		void SetPlayerMapPointer(map< int, SPlayerObj >* pmPlayers) { m_pmPlayers = pmPlayers; };

		/// Index des Startfeldes setzen (Multiplayer only!)
		void SetPlayerStartPos(int nStartPos) { m_nPlayerStartPos = nStartPos; }

		/// Level-Datei einlesen
		/// @param szFile	Datei die geladen werden soll
		/// @param bInitPlayer	Spieler auf Startpunkt setzen?
		/// @return Laden erfolgreich?
		bool LoadFromFile(char* szFile, bool bInitPlayer = true);

		/// Leveldatei neu laden
		/// @return Laden erfolgreich?
		bool Reload();

		/// Aktualisiert die Position und Blickrichtung des Spielers
		void Update();

		/// Zeichnen der Welt
		void Draw();

		/// Zeichne transparente Objekte (diese muessen als letztes mit
		/// aktiviertem DEPTH_BUFFER in umgekehrter Reihenfolge zum
		/// Betrachterstandpunkt gezeichnet werden!)
		void DrawTransparentObjects();

		/// Kollidiert der Spieler mit einem Block oder einem anderen Spieler
		/// oder ist er ausserhalb des Levels?
		/// @return bool true = Spieler kollidiert, sonst false
		bool PlayerCollides(ECollision* peCollSide1, ECollision* peCollSide2, int* pnCollPlayerID);

		/// Bei Kollision mit dem Boden eines Start- oder Zielfeldes findet eine
		/// Landung statt
		/// @return bool true = Spieler landet, sonst false
		bool PlayerLanded();

		/// Spieler hat das Ziel erreicht?
		/// @return bool true = Ziel erreicht, sonst false
		bool PlayerReachesGoal();

		/// Spieler ist ueber Spezialfeld?
		bool PlayerReachesSpecialField(const int nSpecialType);

		/// Kamera und versch. Levelwerte initialisieren
		bool SetupLevel();

		/// Spielerposition bei Kollision zurücksetzen? Beim Multiplayer nötig!
		void SetAdjustPlayer(bool b) { m_bAdjustPlayer = b; }

		/// Spielerposition initialisieren: Spieler auf einen Block setzen
		void SetPlayerOnBlock(int nBlockX, int nBlockY, int nBlockZ);

		/// Zeiger auf den aktuellen Block zurueck geben
		SBlock* GetCurrBlock() { return m_pCurrBlock; };
		SBlock* GetCurrBlock(int* x, int* y, int* z)
		{
			*x = m_nPlayerBlockX;
			*y = m_nPlayerBlockY;
			*z = m_nPlayerBlockZ;

			return m_pCurrBlock;
		}

		/// Kamera-Distanz bekommen.
		float GetCamDistance() { return m_fCamDistance; };

		/// Kamera-Distanz setzen
		void SetCamDistance(float fCamDistance)
		{
			m_fCamDistance = fCamDistance;
			if (m_fCamDistance > CAM_MAX_DISTANCE)
				m_fCamDistance = CAM_MAX_DISTANCE;
			if (m_fCamDistance < CAM_MIN_DISTANCE)
				m_fCamDistance = CAM_MIN_DISTANCE;
		};

		/// Winkel holen
		float GetCamAngle() { return m_fCamAngle; }

	private:
		/// Levelspeicher freigeben
		void Clear();

		/// Das gesamte Level zeichnen
		void DrawWorld();

		/// CallList fuer die SkyBox erstellen
		void GenSkyBox();

		/// Zeichnen der Skybox
		void DrawSkyBox();

		/// Ueberpruefen auf Kollision einer bestimmten Seite oder auf Kollision mit
		/// anderem Spieler in diesem Block
		/// @return -1 wenn keine Kollision, 0 bei Kollision mit Block, > 0 bei Spieler-ID
		///			des Spielers, mit dem man kollidiert ist
		int PlayerSideCollides(const int nBlockXOffset, const int nBlockYOffset, const int nBlockSide, const int nBlockSide2 = BLOCK_NOSIDE);

		/// Testet auf Spieler-Spieler Kollision auf dem angegebenen Feld
		/// @return Spielerobjekt des Spielers, mit dem kollidiert wurde oder -1 wenn keine Kollision
		SPlayerObj* PlayerVsPlayerCollision(int nTestFieldX, int nTestFieldY);

		/// Ueberprueft, ob Kollisions-"Box" mit einer Seite oder gleich zwei Seiten
		/// des nebenstehenden Blocks kollidiert
		/// @return Nummer des Eckpunktes oder -1 wenn keine Kollision
		int PlayerBoxCollides(const int nBlockSide1, const int nBlockSide2 = BLOCK_NOSIDE);

		/// Ueberprueft, ob ein Eckpunkt der Kollisions-"Box" mit einer Seite oder gleich zwei Seiten
		/// des nebenstehenden Blocks kollidiert
		inline bool PlayerBoxPointCollides(CVec2D* pvPoint, const int nBlockSide);

		/// Spielerposition zurücksetzen, damit der Spieler nicht plötzlich hinter der Wand
		/// steckt
		inline void AdjustPlayer(CVec2D* pvPoint, const int nBlockSide);

		/// Zeiger auf einen Block zurueckgeben
		/// @param nBlockX X-Koordinate der Blockmatrix
		/// @param nBlockY Y-Koordinate der Blockmatrix
		/// @param nBlockZ Z-Koordinate der Blockmatrix
		/// @return SBlock* Zeiger auf Block
		///					Falls Block nicht existiert (ausserhalb der Blockdimensionen): NULL
		inline SBlock* GetBlock(const int nBlockX, const int nBlockY, const int nBlockZ);

		/// Bloecke herausfinden, die das Sichtfeld beeintraechtigen. Diese
		/// werden im Array fuer transparente Bloecke gespeichert, da transparente
		/// Objekte am Ende gezeichnet werden muessen!
		/// @see m_arTransBlocks
		void FindTransparentBlocks();

		/// Findet den aktuellen Block, über dem der Spieler mit den
		/// uebergebenen Koordinaten gerade steht
		inline void FindPlayerBlock(const float fPosX, const float fPosY, const float fPosZ, int* pnBlockX, int* pnBlockY, int* pnBlockZ);

		/// Einen Block zeichnen
		/// @param iBlockX X-Koord. der Blockmatrix
		/// @param iBlockY Y-Koord. der Blockmatrix
		/// @param iBlockZ Z-Koord. der Blockmatrix
		/// @param Transparente Bloecke zeichnen?
		void DrawBlock(int iBlockX, int iBlockY, int iBlockZ, bool bDrawTransp = false);

		/// Bewegung und Blick realisieren, Tasten abfangen
		void ProcessMoving();

		/// Kamerabewegung realisieren
		void MoveCam();

		vector< vector< vector<SBlock> > > m_Blocks;	///< Bloecke des Levels in einem multidim. Vektor
		SBlock* m_pCurrBlock;							///< Zeiger auf den Block, ueber dem der Spieler gerade steht
		map< int, SPlayerObj >* m_pmPlayers;				///< Zeiger auf Vektor von Spielern im Multiplayermodus für Inter-Spieler-Kollision

		int m_arTransBlocks[256][3];				///< Array von Indizes transparenter Bloecke @see DrawTransparentObjects
		int m_nTransBlockCounter;					///< Anzahl der transparenten Bloecke im Array m_arTransBlocks

		char m_szFile[256];						///< Leveldatei

		CVec3D m_vCamPos;		///< Kameraposition
		float m_fCamDistance;	///< Kamerazoom
		float m_fCamAngle;		///< Winkel der angibt, wie hoch die Kamera ueber dem Objekt steht

		SPlayerObj* m_pPlayerObj;	///< Zeiger auf die aktuelle Spielerinstanz

		bool m_bAdjustPlayer;		///< Beim Multiplayer: Spielerposition nach der Kollision anpassen, so dass er sich nicht verkeilen kann
		
		int m_nPlayerBlockX;	///< X-Koord. des Blocks, ueber dem sich der der Spieler gerade befindet
		int m_nPlayerBlockY;	///< Y-Koord. des Blocks, ueber dem sich der der Spieler gerade befindet
		int m_nPlayerBlockZ;	///< Z-Koord. des Blocks, ueber dem sich der der Spieler gerade befindet
		//int m_nStartBlockX;	///< X-Koord. des Blocks, ueber dem sich der der Startpunkt befindet
		//int m_nStartBlockY;	///< Y-Koord. des Blocks, ueber dem sich der der Startpunkt befindet
		//int m_nStartBlockZ;	///< Z-Koord. des Blocks, ueber dem sich der der Startpunkt befindet
		vector< SBlockCoord > m_vStartBlocks;	/// Vektor mit Koord. der Startbloecke

		int m_nPlayerStartPos;	///< Startpositionsindex dieses Spielers
		int m_nEndBlockX;		///< X-Koord. des Blocks, ueber dem sich der der Zielpunkt befindet
		int m_nEndBlockY;		///< Y-Koord. des Blocks, ueber dem sich der der Zielpunkt befindet
		int m_nEndBlockZ;		///< Z-Koord. des Blocks, ueber dem sich der der Zielpunkt befindet

		float m_fFuelAnimAngle;	///< Fuer die Dreh-Animation der Treibstoffkanister

		GLuint m_nSkyBoxCallList;	///< Call Liste fuer die Skybox
};

#endif
