#include "GameManager.hpp"
#include "ServerGame.hpp"
#include "../tools/FileTools.hpp"
#include <cstring>

CGameManager::CGameManager(bool bMultiGames, char* szMapDir)
{
	m_bMultiGames = bMultiGames;
	m_pMutex = SDL_CreateMutex();
	
	strcpy(m_szMapDir, szMapDir);
	ReadMaps();
}

bool CGameManager::Enter(CServerClient* pClient)
{
	// Es gibt keinen Server-Raum in dem sich Spieler aufhalten
	// im LAN-Spiel-Modus
	if (m_bMultiGames == false)
	{
		return false;
	}
	
	for (u16 i = 0; i < m_vpPlayers.size(); i++)
	{
		if (m_vpPlayers[i] == pClient)
		{
			// Client ist schon im Server-Raum
			return true;
		}
	}
	
	SDL_LockMutex(m_pMutex);
	DEBUG_PRINTL("Client " << pClient->GetInfo()->nID << " entered server room");
	m_vpPlayers.push_back(pClient);
	SDL_UnlockMutex(m_pMutex);
	return true;
}

void CGameManager::Leave(CServerClient* pClient)
{
	SDL_LockMutex(m_pMutex);
	std::vector<CServerClient*>::iterator iter;
	for (iter = m_vpPlayers.begin(); iter != m_vpPlayers.end(); iter++)
	{
		if (*iter == pClient)
		{
			m_vpPlayers.erase(iter);
			DEBUG_PRINTL("Client " << pClient->GetInfo()->nID << " left server room");
			break;
		}
	}
	SDL_UnlockMutex(m_pMutex);
}

u16 CGameManager::CountPlayer()
{
	return m_vpPlayers.size();
}

void CGameManager::GetPlayer(SPlayerInfo* pPlayerInfos)
{
	SDL_LockMutex(m_pMutex);
	CServerClient* pPlayer;
	for (u16 i = 0; i < m_vpPlayers.size(); i++)
	{
		pPlayer = m_vpPlayers[i];
		memcpy(pPlayerInfos + i, pPlayer->GetInfo(), sizeof(SPlayerInfo));
	}
	SDL_UnlockMutex(m_pMutex);
}

CServerGame* CGameManager::CreateGame(char* szName, u8 nMaxPlayers, 
		bool bAllowLateJoin, bool bAllowVisitors, char* szPassword)
{
	// Nur ein Spiel erlauben wenn m_bMultiGames auf false gesetzt ist
	if (m_bMultiGames == false && m_mpGames.size() >= 1)
	{
		return NULL;
	}
	
	CServerGame* pGame = new CServerGame(szName, nMaxPlayers, bAllowLateJoin, 
			bAllowVisitors, szPassword);
	
	// Spiel registrieren
	SDL_LockMutex(m_pMutex);
	m_mpGames[pGame->GetGameInfo()->nID] = pGame;
	SDL_UnlockMutex(m_pMutex);
	
	return pGame;
}

bool CGameManager::DeleteGame(u16 nGameID)
{
	std::map<u16, CServerGame*>::iterator iter = m_mpGames.find(nGameID);
	if (iter == m_mpGames.end())
	{
		return false;
	}
	else
	{
		// Delete beim ServerGame aufrufen um Teilnehmer zu informieren
		iter->second->Delete();
		
		SDL_LockMutex(m_pMutex);
		m_mpGames.erase(iter);
		SDL_UnlockMutex(m_pMutex);
		return true;
	}
}

bool CGameManager::JoinGame(u16 nGameID, CServerClient* pPlayer, char* szPassword)
{
	CServerGame* pGame = GetGame(nGameID);
	if (pGame == NULL)
	{
		return false;
	}
	else
	{
		DEBUG_PRINTL("Client " << pPlayer->GetInfo()->nID << " joined game " << nGameID);
		return pGame->Join(pPlayer, szPassword);
	}
}

bool CGameManager::VisitGame(u16 nGameID, CServerClient* pPlayer)
{
	CServerGame* pGame = GetGame(nGameID);
	if (pGame == NULL)
	{
		return false;
	}
	else
	{
		return pGame->Visit(pPlayer);
	}
} 
 
void CGameManager::LeaveGame(u16 nGameID, u16 nPlayer)
{
	CServerGame* pGame = GetGame(nGameID);
	if (pGame != NULL)
	{
		pGame->Leave(nPlayer);
		DEBUG_PRINTL("Client " << nPlayer << " left game " << nGameID);
		if (pGame->GetGameInfo()->nNumPlayers == 0)
		{
			// Wenn keine Spieler mehr teilnehmen wird das Spiel beendet und
			// geloescht
			DeleteGame(nGameID);
			DEBUG_PRINTL("No players left - Game deleted");
		}
	}
}

u16 CGameManager::CountGames(bool bOnlyFreeGames)
{
	if (bOnlyFreeGames == false)
	{
		return m_mpGames.size();
	}
	else
	{
		u16 nNumGames = 0;
		std::map<u16, CServerGame*>::iterator iter;
		SGameInfo *pInfo;
		CServerGame* pGame;
		
		for(iter = m_mpGames.begin(); iter != m_mpGames.end(); iter++)
		{
			pGame = iter->second;
			pInfo = pGame->GetGameInfo();
			if (pInfo->nNumPlayers < pInfo->nMaxPlayers && 
					(pInfo->bLateJoin == true || pInfo->eState == SERVERGAME_CONFIG))
			{
				nNumGames++;
			}
		}
		return nNumGames;
	}
}

CServerGame* CGameManager::GetGame(u16 nGameID)
{
	if (m_mpGames.find(nGameID) == m_mpGames.end())
	{
		return NULL;
	}
	else
	{
		return m_mpGames.find(nGameID)->second;
	}
}

void CGameManager::GetGames(bool bOnlyFreeGames, SGameInfo* pGameInfos)
{
	GetGames(bOnlyFreeGames, m_mpGames.size(), 0, pGameInfos);
}

void CGameManager::GetGames(bool bOnlyFreeGames, u16 nNumGames, 
		u16 nOffset, SGameInfo* pGameInfos)
{
	std::map<u16, CServerGame*>::iterator iter;
	SGameInfo *pBuffer = pGameInfos, *pInfo;
	CServerGame* pGame;
	
	iter = m_mpGames.begin();
	/// Move the iter to the chosen position
	if (nOffset != 0)
	{
		for(iter = m_mpGames.begin(); iter != m_mpGames.end(); iter++)
		{
			pGame = iter->second;
			pInfo = pGame->GetGameInfo();
			
			if (bOnlyFreeGames == false || 
				(pInfo->nNumPlayers < pInfo->nMaxPlayers && 
					(pInfo->bLateJoin == true || pInfo->eState == SERVERGAME_CONFIG)))
			{
				if (--nOffset == 0)
					break;
			}
		}
	}
	
	for (; iter != m_mpGames.end(); iter++)
	{
		pGame = iter->second;
		pInfo = pGame->GetGameInfo();
		if (bOnlyFreeGames == false || 
				(pInfo->nNumPlayers < pInfo->nMaxPlayers && 
				(pInfo->bLateJoin == true || pInfo->eState == SERVERGAME_CONFIG)))
		{
			memcpy(pBuffer, pInfo, sizeof(SGameInfo));
			pBuffer++;
			
			if (--nNumGames == 0)
				break;
		}
	}
}

u16 CGameManager::CountMaps()
{
	return m_mMaps.size();
}

bool CGameManager::GetMap(u8 nMapID, std::string* pstrFilename)
{
	std::map<u8, SMapInfo>::iterator iter;
	iter = m_mMaps.find(nMapID);
	
	if (iter == m_mMaps.end())
	{
		return false;
	}
	
	pstrFilename->clear();
	pstrFilename->append(m_szMapDir);
	pstrFilename->append("/");
	pstrFilename->append(iter->second.szFile);
	
	return true;
}

void CGameManager::GetMaps(SMapInfo* pMapInfos)
{
	std::map<u8, SMapInfo>::iterator iter;
	u16 nCounter = 0;
	for (iter = m_mMaps.begin(); iter != m_mMaps.end(); iter++)
	{
		memcpy(pMapInfos + nCounter, &iter->second, sizeof(SMapInfo));
		nCounter++;
	}
}

void CGameManager::Chat(CServerClient* pSender, char* szMessage)
{
	std::vector<CServerClient*>::iterator iter;
	CServerClient* pClient;
	SDL_LockMutex(m_pMutex);
	for (iter = m_vpPlayers.begin(); iter != m_vpPlayers.end(); iter++)
	{
		pClient = *iter;
		pClient->Chat(pSender->GetInfo()->nID, szMessage);
	}
	SDL_UnlockMutex(m_pMutex);
}

void CGameManager::ReadMaps()
{
	std::vector<string> vFiles;
	SMapInfo sMap;
	string szMapName;
	string strMapDir(m_szMapDir);
	 

#ifdef WIN32
	strMapDir.append("*.txt");
	const char* szDirWowWindowsWowWorkaround = strMapDir.c_str();	//WOW! http://msdn.microsoft.com/en-us/library/aa365745(VS.85).aspx
	int nOffset = 6;
#else
	const char* szDirWowWindowsWowWorkaround = m_szMapDir;
	int nOffset = 5;
#endif
	
	if (CFileTools::GetFiles(szDirWowWindowsWowWorkaround, vFiles))
	{
		for (u8 i = 0; i < vFiles.size(); i++)
		{
			memset(&sMap, 0, sizeof(SMapInfo));
			
			sMap.nID = i;
			sMap.nNumPlayer = (u8) atoi(&(vFiles.at(i)[0]));
			memcpy(sMap.szFile, vFiles.at(i).c_str(), vFiles.at(i).length());
			szMapName = vFiles.at(i).substr(1, vFiles.at(i).length() - nOffset);
			memcpy(sMap.szName, szMapName.c_str(), szMapName.length());
			
			DEBUG_PRINTL((u16) sMap.nNumPlayer << " " << sMap.szName << " " << sMap.szFile);
			
			m_mMaps[i] = sMap;
		}
	}
}

