#ifndef GAMEMANAGER_HPP_
#define GAMEMANAGER_HPP_

#include <vector>
#include <map>
#include "../net/SocketFramework.hpp"
#include "../net/NetworkTypes.hpp"
#include "ServerClient.hpp"

class CServerGame;

/// Spieleverwaltung

class CGameManager
{
	public:
		/// Konstuktor
		/// @param	bMultiGames	Unterstützung für Verwaltung mehrerer Spiele
		/// @param	szMapDir	Verzeichnis in dem sich Map-Daten befinden

		CGameManager(bool bMultiGames, char* szMapDir);
		
		/// Tritt dem Server-Raum bei, der Ort in dem Spieler sich aufhalten
		/// wenn sie noch keinem Spiel beigetreten sind
		/// @param	pClient	CServerClient-Instanz der beitreten soll
		/// @return	true wenn beigetreten werden konnte sonst false

		bool Enter(CServerClient* pClient);
		
		/// Verlaesst den Server-Raum
		/// @param	pClient	CServerClient-Instanz die den Raum verlaesst

		void Leave(CServerClient* pClient);
		
		/// Gibt die Anzahl Spieler zurück die noch keinem Spiel beigetreten sind
		/// @return	Anzahl Spieler

		u16 CountPlayer();
		
		/// Schreibt Spielerinformationen (SPlayerInfo) jedes Spielers
		/// in einen Puffer
		/// @param	pPlayerInfos	Der Zielpuffer

		void GetPlayer(SPlayerInfo* pPlayerInfos);
		
		/// Erstellt ein neues Spiel mit Passwortschutz und gibt
		/// die Spiel-ID zurück
		/// @param	szName	Name des Spiels
		/// @param	nMaxPlayers	Maximale Anzahl Spieler
		/// @param	bAllowLateJoin	Erlaube das Beitreten von Spielern nachdem
		/// 							das Spiel gestartet wurde?
		/// @param	bAllowVisitors	Zuschauer erlauben?
		/// @param	szPassword	Das Passwort (NULL für kein Passwort)
		/// @return	Einen Zeiger auf das neu angelegte Spiel

		CServerGame* CreateGame(char* szName, u8 nMaxPlayers, 
				bool bAllowLateJoin, bool bAllowVisitors, char* szPassword);
		
		/// Entfernt ein Spiel vom Game Manager
		/// @param	nGameID	Die Spiel-ID
		/// @return	true wenn das Spiel entfernt wurde

		bool DeleteGame(u16 nGameID);
		
		/// Tritt einem Spiel bei
		/// @param	nGameID	Die Spiel-ID
		/// @param	pPlayer	Eine Instanz der Client-Kommunikation
		/// @param	szPassword	Passwort zum beitritt
		/// @return	true wenn beigetreten werden konnte sonst false

		bool JoinGame(u16 nGameID, CServerClient* pPlayer, char* szPassword);
		
		/// Tritt einem Spiel als Zuschauer bei
		/// @param	nGameID	Die Spiel-ID
		/// @param	pPlayer	Eine Instanz der Client-Kommunikation
		/// @return	true wenn zugeschaut werden kann sonst false

		bool VisitGame(u16 nGameID, CServerClient* pPlayer);
		
		/// Verlaesst ein Spiel
		/// @param	nGameID	Die Spiel-ID
		/// @param	nPlayer Zuschauer oder Mitspieler der das Spiel verl�sst

		void LeaveGame(u16 nGameID, u16 nPlayer);
		
		/// Schreibt Spielerinformationen (SPlayerInfo) jedes Spielers
		/// des Spiels in einen Puffer
		/// @param	nGameID	Die Spiel-ID
		/// @param	pPlayerInfos	Der Zielpuffer
		/// @return	true wenn das Spiel existiert sonst false

		bool GetGamePlayer(u16 nGameID, SPlayerInfo* pPlayerInfos);
		
		/// Gibt die Anzahl registrierter oder verfügbarer Spiele zurück
		/// @param	bOnlyFreeGames	Wenn true werden nur Spiel gezählt die nicht
		/// 							voll besetzt sind und entweder noch nicht
		/// 							gestartet wurden oder das Beitreten nach
		/// 							Spielstart erlaubt ist
		/// @return	Anzahl verfügbarer Spiele

		u16 CountGames(bool bOnlyFreeGames);
		
		/// Gibt die Spielinformationen zu einem Spiel zurück
		/// @param	nGameID	Die Spiele-ID
		/// @return	Spieleinformationen (NULL wenn ein
		/// 			Spiel mit der ID nicht existierT)
 
		CServerGame* GetGame(u16 nGameID);
				
		/// Gibt die Anzahl registrierter oder verfügbarer Spiele zurück
		/// @param	bOnlyFreeGames	Wenn true werden nur Spiel gezählt die nicht
		/// 							voll besetzt sind und entweder noch nicht
		/// 							gestartet wurden oder das Beitreten nach
		/// 							Spielstart erlaubt ist
		/// @param	pGameInfos	Puffer in den die Spiele-Informationen
		/// 						geschrieben wird

		void GetGames(bool bOnlyFreeGames, SGameInfo* pGameInfos);
		
		/// Gibt die Anzahl registrierter oder verfügbarer Spiele zurück
		/// Dabei kann die Anzahl auszulesender Spiele angegeben werden
		/// @param	bOnlyFreeGames	Wenn true werden nur Spiel gezählt die nicht
		/// 							voll besetzt sind und entweder noch nicht
		/// 							gestartet wurden oder das Beitreten nach
		/// 							Spielstart erlaubt ist
		/// @param	nNumGames	Anzahl auszulesender Spiele
		/// @param	nOffset	Offset
		/// @param	pGameInfos	Puffer in den die Spiele-Informationen
		/// 						geschrieben wird

		void GetGames(bool bOnlyFreeGames, u16 nNumGames, u16 nOffset, 
				SGameInfo* pGameInfos);
		
		/// Gibt die Anzahl verfügbarer Maps
		/// @return	Anzahl verfügbarer Maps

		u16 CountMaps();
		
		/// Gibt den Dateinamen zu einer Map zurueck
		/// @param	nMapID	ID der Map
		/// @param	pstrFilename	Zeiger auf einen String in dem der Dateiname gespeichert wird
		/// @return	true wenn Map gefunden wurde sonst false

		bool GetMap(u8 nMapID, std::string* pstrFilename);
		
		/// Schreibt Map-Informationen in einen Puffer
		/// @param	pMapInfos	Puffer in den Karteninformationen kopiert werden

		void GetMaps(SMapInfo* pMapInfos);
		
		///pBuffer
		/// Liest alle Daten einer Map aus deren Datei ein
		/// @param	szMapName	Name der Map
		/// @param	pLength	Speicher auf dem die Größe der Datei
		/// 			geschrieben wird
		/// @return	Einen Zeiger auf die Daten der Datei (Speicher
		/// 			muss danach wieder freigegeben werden

		u8* GetMapData(char* szMapName, u16* pLength);
		
		/// Schickt eine Nachricht an alle Spieler
		/// die noch keinem Spiel beigetreten sind
		/// @param	pSender	referenz auf sendenden client
		/// @param	szMessage	Zu sendende Nachricht

		void Chat(CServerClient* pSender, char* szMessage);
		
	private:
		/// Unterstütze die Verwaltung mehrerer Spiele
		bool m_bMultiGames;
		
		/// Mutex zum Thread-sicheren Zugriff auf Daten
		SDL_mutex* m_pMutex;
		
		/// Spieler die noch keinem Spiel beigetreten sind
		std::vector<CServerClient*> m_vpPlayers;
			
		/// Informationen zu allen verfügbaren Maps
		std::map<u8, SMapInfo> m_mMaps;
		
		/// Map aller registrierten Spiele
		std::map<u16, CServerGame*> m_mpGames;
		
		/// Verzeichnis in dem Map-Daten liegen
		char m_szMapDir[MAPDIR_LENGTH];
		
		/// Liest Informationen aller verfuegbare Karten
		/// in m_szMapDir

		void ReadMaps();
};

#endif ///GAMEMANAGER_HPP_///

