#include "Server.hpp"
#include <vector>
#include <cstdlib>
#include <ctime>

CServer::CServer(u16 nPort, bool bMultiGames, char* szMapDir)
	: BaseThread()
{
	// Zufallsgenerator initialisieren
	srand(time(NULL));

	m_nPort = nPort;
	/// Anfragen von allen Adressen annehmen
	m_sSocket = NULL;
	m_pMutex = SDL_CreateMutex();
	m_pGameManager = new CGameManager(bMultiGames, szMapDir);
	m_bRunning = false;
}

CServer::~CServer()
{
	Terminate();
	
	SDL_DestroyMutex(m_pMutex);
	delete m_pGameManager;
}

int CServer::Run()
{
	DEBUG_PRINTL("Server started");
	
	if (SDLNet_Init() == -1)
	{
		std::cout << "SDLNet_Init: " << SDLNet_GetError() << std::endl;
	    return -1;
	}
	
	/// Socket erzeugen
	if (m_sSocket != NULL)
		SDLNet_TCP_Close(m_sSocket);
	
	IPaddress sIP = { INADDR_ANY, SDL_Swap16(m_nPort) };
	if ((m_sSocket = SDLNet_TCP_Open(&sIP)) == NULL)
	{
		DEBUG_PRINTL("Error TCP-Open: " << SDLNet_GetError());
		return -1;
	}
		
	if (m_sSocket == NULL)
		return -1;
	
	TCPsocket sClientSocket;
	m_bRunning = true;
	while (m_bTerminate == false)
	{
		sClientSocket = SDLNet_TCP_Accept(m_sSocket);
		if (sClientSocket != NULL)
		{
			DEBUG_PRINTL("New Client connected");
			if (RegisterClient(sClientSocket) == false)
			{
				DEBUG_PRINTL("Error while Registering Client");
				SDLNet_TCP_Close(sClientSocket);
			}
		}
		SDL_Delay(SERVER_REFRESH_INTERVAL);
	}
	
	/// Sockets der Clients beenden
	SDL_LockMutex(m_pMutex);
	int nNumClients = m_vClients.size();
	CServerClient** pClient = new CServerClient*[nNumClients];
	for (u16 i = 0; i < nNumClients; i++)
	{
		pClient[i] = m_vClients[i];
	}
	SDL_UnlockMutex(m_pMutex);
	
	for (u16 i = 0; i < nNumClients; i++)
	{
		pClient[i]->Terminate();
		FREEMEM(pClient[i]);
	}
	FREEMEM_ARRAY(pClient);
	m_vClients.clear();
	
	// Server socket schließen
	SDLNet_TCP_Close(m_sSocket);
	m_sSocket = NULL;
	m_bRunning = false;
	
	return true;
}

bool CServer::IsRunning()
{
	return m_bRunning;
}

bool CServer::RegisterClient(TCPsocket sClientSocket)
{
	CServerClient* pClient = new CServerClient(sClientSocket, this);
	
	/// Neuen Thread erzeugen
	if (pClient->Start() == false)
	{
		DEBUG_PRINTL("Thread could not be created");
		delete pClient;
		
		return false;
	}
	else
	{
		// Thread-Pointer dem Client mitteilen
		SDL_LockMutex(m_pMutex);
		m_vClients.push_back(pClient);
		SDL_UnlockMutex(m_pMutex);
		
		return true;
	}
}

void CServer::UnregisterClient(CServerClient* pClient)
{
	std::vector<CServerClient*>::iterator iter;
	
	/// Wichtig! Clients koennen gleichzeitig beendet werden!
	SDL_LockMutex(m_pMutex);
	for (iter = m_vClients.begin(); iter != m_vClients.end(); iter++)
	{
		if (*iter == pClient)
		{
			m_vClients.erase(iter);
			break;
		}
	}
	SDL_UnlockMutex(m_pMutex);
}

CGameManager* CServer::GetGameManager()
{
	return m_pGameManager;
}