#include "../globals.hpp"
#include "ServerClient.hpp"
#include "ServerGame.hpp"
#include "Server.hpp"
#include <cstdlib>
#include <ctime>

u16 CServerClient::nClientCounter = 1;

CServerClient::CServerClient(TCPsocket psSocket, CServer* pServer)
	: BaseThread()
{
	m_sSocket = psSocket;
	m_pServer = pServer;

	m_pMutex = SDL_CreateMutex();
	m_nAuthToken = 0;
	
	m_bInGame = false;
	m_pCurrentGame = NULL;
	m_sPlayerInfo.nID = CServerClient::nClientCounter++;
}

CServerClient::~CServerClient()
{
	Terminate();
}

void CServerClient::AddMessage(CMessage* psMessage)
{
	SDL_LockMutex(m_pMutex);
	m_vMessageQueue.push_back(psMessage);
	SDL_UnlockMutex(m_pMutex);
}

int CServerClient::Run()
{
	SMessageHeader header;
	
	while(!m_bTerminate)
	{
		// Header empfangen um zu erfahren wieviele Nachrichten anstehen
		if (CMessage::SecureRecv(m_sSocket, (u8*) &header, sizeof(header)) == false)
		{
			DEBUG_PRINTL("Socket closed");
			Terminate();
			continue;
		}
		
		// Richtige Byte-Order herstellen
		header.nAuthID = SDLNet_Read32(&header.nAuthID);
	
		// Wenn Benutzer-Authentifizierung fehlschlaegt wird die Verbindung beendet
		if (header.nAuthID != m_nAuthToken)
		{
			DEBUG_PRINTL("Authentication token mismatch!");
			DEBUG_PRINTL("ClientTokenOnServerSide: " << m_nAuthToken);
			DEBUG_PRINTL("ClientTokenOnClientSide: " << header.nAuthID);
			Terminate();
			continue;
		}
		
		header.nNumMessages = SDLNet_Read32(&header.nNumMessages);
		DEBUG_PRINTL("Number of Messages: " << header.nNumMessages);

		if (HandleMessages(header.nNumMessages, (header.nAuthID == 0)) == false)
		{
			Terminate();
			continue;
		}
		
		// Eine Nachricht mit dem aktuellen Spielefortschritt
		// generieren wenn Spieler an einem Spiel teilnimmt
		if (m_bInGame == true && 
				m_pCurrentGame->GetGameInfo()->eState == SERVERGAME_RUNNING)
		{
			DEBUG_PRINTL("Sending server progress");
			GenerateGameProgressMessage();
		}
		
		if (SendMessageQueue() == false)
		{
			Terminate();
			continue;
		}
	}
	
	DEBUG_PRINTL("Client received termination signal");
	if (m_bInGame == true)
	{
		m_pServer->GetGameManager()->LeaveGame(m_pCurrentGame->GetGameInfo()->nID, m_sPlayerInfo.nID);
		m_bInGame = false;
		m_pCurrentGame = NULL;
	}
		
	m_pServer->GetGameManager()->Leave(this);
	m_pServer->UnregisterClient(this);
	SDLNet_TCP_Close(m_sSocket);
	
	return 0;
}

bool CServerClient::SendMap(u8 nMapID)
{
	std::string strFilename;
	if (m_pServer->GetGameManager()->GetMap(nMapID, &strFilename) == true)
	{
		SResponseGameGetMap sResponse = { { 0, GAME_GET_MAP, CODE_OK }, 0 };
		FILE* fd = fopen(strFilename.c_str(), "r");
		if (fd == NULL)
		{
			DEBUG_PRINTL(strFilename.c_str() << " couldn't be opened");
		}
		u16 nSize;
		fseek(fd, 0, SEEK_END);
		nSize = ftell(fd);
		fseek(fd, 0, SEEK_SET);
		
		DEBUG_PRINTL("File is " << nSize << " Bytes large");
		
		SDLNet_Write16(
			sizeof(SResponseGameGetMap) + nSize, // Gesamtpaketlaenge!!
			&sResponse.sHeader.nMessageLength);
		
		SDLNet_Write16(
			nSize,
			&sResponse.nLength);
		
		CMessage* pMessage = new CMessage(sizeof(SResponseGameGetMap) + nSize, (u8*) &sResponse, sizeof(SResponseGameGetMap));
		u8* pData = pMessage->Data() + sizeof(SResponseGameGetMap);
		
		//while (!feof(fd))
		//{
			int nBytesRead = fread(pData, 1, nSize, fd);
			DEBUG_PRINTL(nBytesRead << " Bytes read");
			//pData += nBytesRead;
		//}
		fclose(fd);
		
		AddMessage(pMessage);
		return true;
	}
	else
	{
		return false;
	}
}

void CServerClient::ProceedLevel(u8 nPlayerStartPos, u32 nStartTime)
{
	SResponseGameNextLevel sResponse = { { 0, GAME_NEXT_LEVEL, CODE_OK }, *m_pCurrentGame->GetGameInfo(), nPlayerStartPos };
	SDLNet_Write32(nStartTime, &sResponse.nStartTime);
	
	SDLNet_Write16(
		sizeof(SResponseGameNextLevel),
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseGameNextLevel), (u8*) &sResponse));
}

bool CServerClient::HandleMessages(u8 nNumMessages, bool bClientNotAuthenticated)
{
	CMessage* pMessage = NULL;
	bool (CServerClient::*pCallback)(CMessage*);
	
	for (u8 i = 0; i < nNumMessages; i++)
	{
		if (pMessage != NULL)
		{
			delete pMessage;
			pMessage = NULL;
		}
		
		pMessage = new CMessage(sizeof(SRequestHeader));
		if (pMessage->Recv(m_sSocket) == false)
		{
			DEBUG_PRINTL("Error receiving request header");
			return false;
		}
		
		SRequestHeader* pHeader = (SRequestHeader*) pMessage->Data();
		// Rest der Nachricht empfangen
		u16 shMessageType = (u16) pHeader->eMessageType; //spaeter auslesen wird nix

		// Test, ob der Client ueberhaupt angemeldet ist, sonst ist nur eine Login-Message erlaubt!
		if (bClientNotAuthenticated && shMessageType != CLIENT_LOGIN)
		{
			DEBUG_PRINTL("Evil client tried sending message of type " << shMessageType << " but did not log in first!");
			DEBUG_PRINTL("Kommst hier net rein, alda!");
			Terminate();
			return false;
		}

		u16 nMessageSize = SDLNet_Read16(&pHeader->nMessageLength);
		DEBUG_PRINTL("Received request with message-length of " << nMessageSize << " Bytes");
		pMessage->Resize(nMessageSize);	//die Headergroesse ist schon mit drin in nMessageSize!
		if (pMessage->Recv(m_sSocket, sizeof(SRequestHeader)) == false)
		{
			DEBUG_PRINTL("Error receiving request");
			return false;
		}
		
		DEBUG_PRINTL("Handling request of type: " << shMessageType);
		switch(shMessageType)
		{
			case CLIENT_LOGIN:
				pCallback = &CServerClient::HandleClientLogin;
				bClientNotAuthenticated = false;
				break;
			
			case CLIENT_LOGOUT:
				pCallback = &CServerClient::HandleClientLogout;
				break;
			
			case CLIENT_FETCH_MESSAGES:
				/// Nachrichten-Queue wird auf jeden fall gesendet
				pCallback = NULL;
				break;
			
			case CLIENT_SYNC:
				pCallback = &CServerClient::HandleClientSync;
				break;
				
			case CLIENT_CHAT:
				pCallback = &CServerClient::HandleClientChat;
				break;
				
			case SERVER_CREATE_GAME:
				pCallback = &CServerClient::HandleServerCreateGame;
				break;
				
			case SERVER_COUNT_GAMES:
				pCallback = &CServerClient::HandleServerCountGames;
				break;
			
			case SERVER_GET_GAMES:
				pCallback = &CServerClient::HandleServerGetGames;
				break;
				
			case SERVER_GET_GAME:
				pCallback = &CServerClient::HandleServerGetGame;
				break;
			
			case SERVER_GET_PLAYER:
				pCallback = &CServerClient::HandleServerGetPlayer;
				break;
				
			case SERVER_GET_MAPS:
				pCallback = &CServerClient::HandleServerGetMaps;
				break;
			case GAME_GET_MAP:
				pCallback = &CServerClient::HandleGameGetMap;
				break;
			case GAME_MAP_CONFIG:
				pCallback = &CServerClient::HandleGameMapConfig;
				break;
				
			case GAME_START:
				pCallback = &CServerClient::HandleGameStart;
				break;
			
			case GAME_JOIN:
				pCallback = &CServerClient::HandleGameJoin;
				break;
							
			case GAME_LEAVE:
				pCallback = &CServerClient::HandleGameLeave;
				break;
			
			case GAME_PLAYER_DIE:
				pCallback = &CServerClient::HandleGamePlayerDie;
				break;
				
			case GAME_PUSH_PROGRESS:
				pCallback = &CServerClient::HandleGamePushProgress;
				break;
							
			case GAME_GET_PLAYER:
				pCallback = &CServerClient::HandleGameGetPlayer;
				break;
				
			case GAME_GET_PLAYERINFO:
				pCallback = &CServerClient::HandleGameGetPlayerInfo;
				break;
			case GAME_NEXT_LEVEL:
				pCallback = &CServerClient::HandleGameNextLevel;
				break;
				
			default:
				// false wenn ung�ltige Anfagen gesendet werden
				delete pMessage;
				return false;
		}
		
		if (pCallback != NULL)
		{
			if ((*this.*pCallback)(pMessage) == false)
			{
				delete pMessage;
				return false;
			}
		}
	}
	
	return true;
}

bool CServerClient::HandleClientLogin(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling CLIENT_LOGIN");
	SRequestClientLogin* psRequest = (SRequestClientLogin*) pMessage->Data();
	memcpy(m_sPlayerInfo.szName, psRequest->szName, PLAYER_NAME_LENGTH);
	
	// Neue Auth-ID in der Antwortnachricht mitteilen
	srand(time(NULL)); //Ohne Dieses gibts IMMER 41!! 
	m_nAuthToken = (u32) rand();
	
	SResponseClientLogin sResponse = { { 0, CLIENT_LOGIN, CODE_OK }, 0, 0 };
	SDLNet_Write16(
			sizeof(SResponseClientLogin), // Gesamtpaketlaenge!!
			&sResponse.sHeader.nMessageLength);
	SDLNet_Write32(m_nAuthToken, &sResponse.nAuthID);
	SDLNet_Write16(m_sPlayerInfo.nID, &sResponse.nID);
	
	DEBUG_PRINTL("Registered player " << m_sPlayerInfo.szName << " with AuthToken " << m_nAuthToken);
	
	m_pServer->GetGameManager()->Enter(this);
	
	AddMessage(new CMessage(sizeof(SResponseClientLogin), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleClientLogout(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling CLIENT_LOGOUT");	
	if (m_bInGame == true)
	{
		m_pServer->GetGameManager()->
			LeaveGame(m_pCurrentGame->GetGameInfo()->nID, m_sPlayerInfo.nID);
	}
	
	SResponseClientLogout sResponse = { { 0, CLIENT_LOGOUT, CODE_OK } };
	AddMessage(new CMessage(sizeof(SResponseClientLogout), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleClientChat(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling CLIENT_CHAT");
	SRequestClientChat* psRequest = (SRequestClientChat*) pMessage->Data();
	
	// Ist der Client gerade in einem Spiel wird die Nachricht
	// an alle Mitspieler gesendet. Wenn nicht geht die Nachricht
	// an alle Clients die ebenfalls noch keinem Spiel beigetreten sind
	if (m_bInGame == true)
	{
		m_pCurrentGame->Chat(this, psRequest->szMessage);
	}
	else
	{
		m_pServer->GetGameManager()->Chat(this, psRequest->szMessage);
	}
	
	// CLIENT_CHAT-Requests erfordern kein Response
	return true;
}

bool CServerClient::HandleClientSync(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling CLIENT_SYNC");
	SResponseClientSync sResponse = { { 0, CLIENT_SYNC, CODE_UNUSED }, 0 };
	SDLNet_Write32(time(NULL), &sResponse.nServerTime);
	
	SDLNet_Write16(
			sizeof(SResponseClientSync), 
			&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseClientSync), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleServerCreateGame(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling SERVER_CREATE_GAME");
	
	SRequestServerCreateGame* psRequest = (SRequestServerCreateGame*) pMessage->Data();
	
	// Antwort-Nachricht erstellen
	// Feld für Return-Code und Daten auf 0 setzen
	SResponseServerCreateGame sResponse = { { 0, SERVER_CREATE_GAME, CODE_UNUSED }, 0 };
	
	// Wenn Spieler schon bei einem Spiel beteiligt ist, kann er
	// kein neues erstellen
	if (m_bInGame == true)
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_INGAME;
	}
	else
	{
		CServerGame* pGame = 
			m_pServer->GetGameManager()->CreateGame(psRequest->szGameName, 
			psRequest->nMaxPlayer, psRequest->bAllowLateJoin, 
			psRequest->bAllowVisitors, psRequest->szPassword);
	
		if (pGame != NULL)
		{
			u16 nGameID = pGame->GetGameInfo()->nID;
			if (m_pServer->GetGameManager()->
					JoinGame(nGameID, this, psRequest->szPassword) == true)
			{
				// Server-Raum verlassen da nun InGame
				m_pServer->GetGameManager()->Leave(this);
			
				// Merken bei welchem Spiel der Client gerade beteiligt ist	
				m_bInGame = true;
				m_pCurrentGame = pGame;
				
				// Alles OK
				sResponse.sHeader.eReturnCode = CODE_OK;
				SDLNet_Write16(pGame->GetGameInfo()->nID, &sResponse.nGameID);
			}
			else
			{
				DEBUG_PRINTL("Error while joining game");
				sResponse.sHeader.eReturnCode = CODE_FAILURE;
			}
		}
		else
		{
			DEBUG_PRINTL("Error while creating game");
			// Fehler-Flag in Antwortnachricht setzen wenn Spiel nicht
			// erstellt werden konnte
			sResponse.sHeader.eReturnCode = CODE_FAILURE;
		}
	}
	
	SDLNet_Write16(
		sizeof(SResponseServerCreateGame), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseServerCreateGame), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleServerCountGames(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling SERVER_COUNT_GAMES");
	
	SRequestServerCountGames* psRequest = (SRequestServerCountGames*) pMessage->Data();
	SResponseServerCountGames sResponse = { { 0, SERVER_COUNT_GAMES, CODE_OK }, 0 };
	u16 nNumGames = m_pServer->GetGameManager()->CountGames(psRequest->bOnlyFreeGames);
	SDLNet_Write16(nNumGames, &sResponse.nNumGames);
	
	SDLNet_Write16(
		sizeof(SResponseServerCountGames), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SRequestServerCountGames), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleServerGetGames(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling SERVER_GET_GAMES");
	
	SRequestServerGetGames* psRequest = (SRequestServerGetGames*) pMessage->Data();

	u16 nNumGames, nOffset = 0;
	if (psRequest->bUseRange == true)
	{
		DEBUG_PRINTL("Use Range");
		nNumGames = SDLNet_Read16(&psRequest->nNumGames);
	}
	else
	{
		DEBUG_PRINTL("Get all games");
		nNumGames = m_pServer->GetGameManager()->CountGames(psRequest->bOnlyFreeGames);
	}
	nOffset = SDLNet_Read16(&psRequest->nOffset);
	DEBUG_PRINTL("Offset: " << nOffset);
	
	// Antwort-Nachricht erstellen
	SResponseServerGetGames sResponse = { { 0, SERVER_GET_GAMES, CODE_UNUSED }, 0 };
	SDLNet_Write16(nNumGames, &sResponse.nNumGames);
	DEBUG_PRINTL("Infos of " << nNumGames << " games");
	
	// Message-Objekt erstellen, das groß genug ist Header und nNumGames *
	// Größe der SGameInfo-Struktur zu speichern
	CMessage* pResponse = new CMessage(
			sizeof(SResponseServerGetGames) + sizeof(SGameInfo) * nNumGames,
			(u8*) &sResponse, sizeof(SResponseServerGetGames));
	
	SGameInfo* pGameInfoBuffer = (SGameInfo*) (pResponse->Data() + sizeof(SResponseServerGetGames));
	m_pServer->GetGameManager()->GetGames(psRequest->bOnlyFreeGames, nNumGames, nOffset, pGameInfoBuffer);
	
	// Byte-Order zum senden anpassen
	CServerClient::CorrectOutgoingByteOrder(pGameInfoBuffer, nNumGames);
	
	SDLNet_Write16(
			pResponse->GetSize(),
			&((SResponseServerGetGames*) pResponse->Data())->sHeader.nMessageLength);
	
	AddMessage(pResponse);
	return true;
}

bool CServerClient::HandleServerGetGame(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling SERVER_GET_GAME");
	
	SRequestServerGetGame* psRequest = (SRequestServerGetGame*) pMessage->Data();
	
	u16 nGameID = SDLNet_Read16(&psRequest->nGameID);
	CServerGame* pGame = m_pServer->GetGameManager()->GetGame(nGameID);
	
	SResponseServerGetGame sResponse;
	sResponse.sHeader.eMessageType = SERVER_GET_GAME;
	if (pGame != NULL)
	{
		 sResponse.sHeader.eReturnCode = CODE_OK;
		 sResponse.sInfo = *pGame->GetGameInfo();
		 
		 // Byte-Order zum senden anpassen
		 CServerClient::CorrectOutgoingByteOrder(&sResponse.sInfo, 1);
	}
	else
	{
		sResponse.sHeader.eReturnCode = CODE_FAILURE;
	}
	
	SDLNet_Write16(
		sizeof(SResponseServerGetGame), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseServerGetGame), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleServerGetPlayer(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling SERVER_GET_PLAYER");
	SResponseServerGetPlayer sResponse = { { 0, SERVER_GET_PLAYER, CODE_UNUSED }, 0 };
	
	u16 nNumPlayer = m_pServer->GetGameManager()->CountPlayer();
	SDLNet_Write16(nNumPlayer, &sResponse.nNumPlayer);
	
	// Message-Objekt erzeugen das genug Speicherplatz besitzt für die
	// Information der einzelnen Spieler
	CMessage* pResponse = new CMessage(
			sizeof(SResponseServerGetPlayer) + sizeof(SPlayerInfo) * nNumPlayer,
			(u8*) &sResponse, sizeof(SResponseServerGetPlayer));
	
	SPlayerInfo* pPlayerInfos = (SPlayerInfo*) (pResponse->Data() + sizeof(SResponseServerGetPlayer));
	m_pServer->GetGameManager()->GetPlayer(pPlayerInfos);
	
	// Byte-Order zum senden anpassen
	CServerClient::CorrectOutgoingByteOrder(pPlayerInfos, nNumPlayer);
	
	SDLNet_Write16(
		pResponse->GetSize(),
		&((SResponseServerGetPlayer*) pResponse->Data())->sHeader.nMessageLength);
	
	AddMessage(pResponse);
	return true;
}

bool CServerClient::HandleServerGetMaps(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling SERVER_GET_MAPS");
	SResponseServerGetMaps sResponse = { { 0, SERVER_GET_MAPS, CODE_UNUSED }, 0 };
		
	u16 nNumMaps = m_pServer->GetGameManager()->CountMaps();
	SDLNet_Write16(nNumMaps, &sResponse.nNumMaps);
	
	// Message-Objekt erzeugen das genug Speicherplatz besitzt für die
	// Information der einzelnen Level
	CMessage* pResponse = new CMessage(
			sizeof(SResponseServerGetMaps) + sizeof(SMapInfo) * nNumMaps,
			(u8*) &sResponse, sizeof(SResponseServerGetMaps));
	
	SMapInfo* pMapInfos = (SMapInfo*) (pResponse->Data() + sizeof(SResponseServerGetMaps));
	m_pServer->GetGameManager()->GetMaps(pMapInfos);
	
	SDLNet_Write16(
		pResponse->GetSize(),
		&((SResponseServerGetMaps*) pResponse->Data())->sHeader.nMessageLength);
	
	AddMessage(pResponse);
	return true;
}

bool CServerClient::HandleGameGetMap(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_GET_MAP");
	SRequestGameGetMap* psRequest = (SRequestGameGetMap*) pMessage->Data();
	SResponseGameGetMap sResponse = { { 0, GAME_GET_MAP, CODE_OK }, 0 };
	
	if (m_bInGame == false)
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_OUTGAME;
	}
	else
	{
		u8 nMapID;
		if (m_pCurrentGame->GetMapIDByLevel(psRequest->nLevel, &nMapID) == false)
		{
			sResponse.sHeader.eReturnCode = CODE_FAILURE;
		}
		else
		{
			if (SendMap(nMapID) == false)
			{
				sResponse.sHeader.eReturnCode = CODE_FAILURE;
			}
			else
			{
				return true;
			}
		}
	}
	
	SDLNet_Write16(
		sizeof(SResponseGameGetMap), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseGameGetMap), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleGameMapConfig(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_MAP_CONFIG");
	SRequestGameMapConfig* psRequest = (SRequestGameMapConfig*) pMessage->Data();
	SResponseGameMapConfig sResponse = { { 0, GAME_MAP_CONFIG, CODE_OK } };
	
	if (m_bInGame == false)
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_OUTGAME;
	}
	else
	{
		if (psRequest->eAction == REQUEST_ADD_MAP)
		{
			for (u8 i = 0; i < psRequest->nAmount; i++)
			{
				if (m_pCurrentGame->AddMap(m_sPlayerInfo.nID, psRequest->nMapID) == false)
				{
					sResponse.sHeader.eReturnCode = CODE_FAILURE;
					break;
				}
			}
		}
		else if (psRequest->eAction == REQUEST_REMOVE_MAP)
		{
			// nAmount wird ignoriert
			if (m_pCurrentGame->RemoveMap(m_sPlayerInfo.nID, psRequest->nMapID) == false)
			{
				sResponse.sHeader.eReturnCode = CODE_FAILURE;
			}
		}
	}
	
	SDLNet_Write16(
		sizeof(SResponseGameMapConfig), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseGameMapConfig), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleGameStart(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_START");
	SResponseGameStart sResponse = { { 0, GAME_START, CODE_OK } };
	
	if (m_bInGame == true)
	{
		if (m_pCurrentGame->Start(m_sPlayerInfo.nID) == false)
		{
			sResponse.sHeader.eReturnCode = CODE_FAILURE;
		}
	}
	else
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_OUTGAME;
	}
	
	
	SDLNet_Write16(
		sizeof(SResponseGameStart), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseGameStart), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleGameJoin(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_JOIN");
	SRequestGameJoin* psRequest = (SRequestGameJoin*) pMessage->Data();
	SResponseGameJoin sResponse = { { 0, GAME_JOIN, CODE_OK } };
	if (m_pCurrentGame != NULL)
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_INGAME;
	}
	else
	{
		u16 nGameID = SDLNet_Read16(&psRequest->nGameID);
		if (m_pServer->GetGameManager()->JoinGame(nGameID, this, psRequest->szPassword))
		{
			m_bInGame = true;
			m_pCurrentGame = m_pServer->GetGameManager()->GetGame(nGameID);
			m_pServer->GetGameManager()->Leave(this);
		}
		else
		{
			sResponse.sHeader.eReturnCode = CODE_FAILURE;
		}
	}
	
	SDLNet_Write16(
		sizeof(SResponseGameJoin), 
		&sResponse.sHeader.nMessageLength);
	
	AddMessage(new CMessage(sizeof(SResponseGameJoin), (u8*) &sResponse));
	return true;
}

bool CServerClient::HandleGameLeave(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_LEAVE");
	
	if (m_bInGame == true)
	{
		m_pServer->GetGameManager()->LeaveGame(m_pCurrentGame->GetGameInfo()->nID, m_sPlayerInfo.nID);
		
		m_pServer->GetGameManager()->Enter(this);
		m_bInGame = false;
		m_pCurrentGame = NULL;
	}
	// Keine Antwort-Nachricht
	return true;
}

bool CServerClient::HandleGamePlayerDie(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_PLAYER_DIE");
	
	if (m_bInGame == true)
	{
		m_pCurrentGame->Die(m_sPlayerInfo.nID);
	}
	
	// Keine Antwort-Nachricht
	return true;
}

bool CServerClient::HandleGamePushProgress(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_PUSH_PROGRESS");
	
	SRequestGamePushProgress* psRequest = (SRequestGamePushProgress*) pMessage->Data();
	if (m_bInGame == true)
	{
		// Byte-Order wiederherstellen
		CServerClient::CorrectIncomingByteOrder(&psRequest->sStats, 1);
		
		m_pCurrentGame->SetPlayerStats(m_sPlayerInfo.nID, &psRequest->sStats);
	}
	
	return true;
}

bool CServerClient::HandleGameGetPlayer(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_GET_PLAYER");
	SResponseGameGetPlayer sResponse = { { 0, GAME_GET_PLAYER, CODE_UNUSED }, 0 };
	CMessage* pResponse;
	
	if (m_bInGame == false)
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_OUTGAME;
		pResponse = new CMessage(sizeof(SResponseGameGetPlayer), (u8*) &sResponse);
	}
	else
	{
		sResponse.sHeader.eReturnCode = CODE_OK;
		u16 nNumPlayer = m_pCurrentGame->GetGameInfo()->nNumPlayers;
		
		SDLNet_Write16(nNumPlayer, &sResponse.nNumPlayer);
	
		// Message-Objekt erzeugen das genug Speicherplatz besitzt für die
		// Information der einzelnen Spieler
		pResponse = new CMessage(
			sizeof(SResponseGameGetPlayer) + sizeof(SPlayerInfo) * nNumPlayer,
			(u8*) &sResponse, sizeof(SResponseGameGetPlayer));
	
		SPlayerInfo* pPlayerInfos = (SPlayerInfo*) (pResponse->Data() + sizeof(SResponseGameGetPlayer));
		m_pCurrentGame->GetPlayer(pPlayerInfos);
		
		// Byte-Order zum senden anpassen
		CServerClient::CorrectOutgoingByteOrder(pPlayerInfos, nNumPlayer);
	}
	
	SDLNet_Write16(
		pResponse->GetSize(),
		&(((SResponseGameGetPlayer*) pResponse->Data())->sHeader.nMessageLength));
	
	AddMessage(pResponse);
	return true;
}

bool CServerClient::HandleGameGetPlayerInfo(CMessage* pMessage)
{
	DEBUG_PRINTL("Handling GAME_GET_PLAYERINFO");
	
	SResponseGameGetPlayerInfo sResponse = { { 0, GAME_GET_PLAYERINFO, CODE_UNUSED }, 0 };
	CMessage* pResponse;
	
	if (m_bInGame == false)
	{
		sResponse.sHeader.eReturnCode = CODE_UNAVAILABLE_OUTGAME;
		pResponse = new CMessage(sizeof(SResponseGameGetPlayer), (u8*) &sResponse);
	}
	else
	{
		sResponse.sHeader.eReturnCode = CODE_OK;
		u16 nNumPlayer = m_pCurrentGame->GetGameInfo()->nNumPlayers;
		DEBUG_PRINTL("Will return " << nNumPlayer << " records");
		SDLNet_Write16(nNumPlayer, &sResponse.nNumPlayer);
	
		// Message-Objekt erzeugen das genug Speicherplatz besitzt für die
		// Information der einzelnen Spieler
		pResponse = new CMessage(
			sizeof(SResponseGameGetPlayerInfo) + (sizeof(SGamePlayerInfo) + sizeof(u16)) * nNumPlayer,
			(u8*) &sResponse, sizeof(SResponseGameGetPlayerInfo));
	
		u8* pOrder = pResponse->Data() + sizeof(SResponseGameGetPlayerInfo);
		u8* pInfos = pOrder + sizeof(u16) * nNumPlayer;
		
		m_pCurrentGame->GetPlayerInfo((u16*) pOrder, (SGamePlayerInfo*) pInfos);
		/*for (int i = 0; i < nNumPlayer; i++)		//ich weiss nicht warum, aber ich musste das Write16 in CServerGame::GetPlayerInfo() reinpacken, damits geht
		{
			SDLNet_Write16(pOrder[i], &pOrder[i]);
		}*/
	}
	
	SDLNet_Write16(
		pResponse->GetSize(),
		&(((SResponseGameGetPlayerInfo*) pResponse->Data())->sHeader.nMessageLength));
	
	AddMessage(pResponse);
	return true;
}

bool CServerClient::HandleGameNextLevel(CMessage* pMessage)
{
	if (m_bInGame == true)
	{
		//Der Spieler, der GAME_NEXT_LEVEL losschickt, hat gewonnen
		m_pCurrentGame->GetSpecificPlayer(m_sPlayerInfo.nID)->sInfo.nRoundsWon++;
		m_pCurrentGame->NextLevel();
	}
	return true;
}

void CServerClient::GenerateGameProgressMessage()
{
	u16 nNumPlayer = m_pCurrentGame->GetGameInfo()->nNumPlayers;
	CMessage* pMessage = new CMessage(sizeof(SResponseGameProgress) +
						sizeof(u16) * nNumPlayer +
						sizeof(SGamePlayerStats) * nNumPlayer);
	
	SResponseGameProgress* psResponse = (SResponseGameProgress*) pMessage->Data();
	u16* pPlayerOrder = (u16*) (pMessage->Data() + sizeof(SResponseGameProgress));
	SGamePlayerStats* pPlayerStats = (SGamePlayerStats*) (pPlayerOrder + nNumPlayer);
	
	psResponse->sHeader.eMessageType = GAME_PROGRESS;
	psResponse->sHeader.eReturnCode = CODE_UNUSED;
	SDLNet_Write16(nNumPlayer, &psResponse->nNumPlayer);
	
	m_pCurrentGame->GetPlayerStats(pPlayerOrder, pPlayerStats);
	// Byte-order korregieren
	for (int i = 0; i < nNumPlayer; i++)
	{
		DEBUG_PRINTL("Player " << i << " is at " << pPlayerStats[i].fPosX << " " << pPlayerStats[i].fPosY << " " << pPlayerStats[i].fPosZ);
		//SDLNet_Write16(pPlayerOrder[i], &pPlayerOrder[i]);	//ist in CServerGame::GetPlayerStats()
	}
	CorrectOutgoingByteOrder(pPlayerStats, nNumPlayer);
	
	SDLNet_Write16(
		pMessage->GetSize(),
		&psResponse->sHeader.nMessageLength);
	
	AddMessage(pMessage);
}

bool CServerClient::SendMessageQueue()
{
	SDL_LockMutex(m_pMutex);
	
	// First send the global message header
	SMessageHeader header = { 0, m_vMessageQueue.size() };
	
	SDLNet_Write32(m_nAuthToken, &header.nAuthID);
	SDLNet_Write32(m_vMessageQueue.size(), &header.nNumMessages);
	
	if (CMessage::SecureSend(m_sSocket, (u8*) &header, sizeof(SMessageHeader)) == false)
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	CMessage* pMessage;
	while (!m_vMessageQueue.empty())
	{
		pMessage = *m_vMessageQueue.begin();
		if (pMessage->Send(m_sSocket) == false)
		{
			SDL_UnlockMutex(m_pMutex);
			return false;
		}
		else
		{
			FREEMEM(pMessage);
			m_vMessageQueue.erase(m_vMessageQueue.begin());
		}
	}
	SDL_UnlockMutex(m_pMutex);
	return true;
}

void CServerClient::Chat(u16 nSender, char* szMessage)
{
	CMessage* pChatMessage = new CMessage(sizeof(SResponseMessageChat));
	SResponseMessageChat* psMessage = (SResponseMessageChat*) pChatMessage->Data();
	
	psMessage->sHeader.eMessageType = MESSAGE_CHAT;
	psMessage->sHeader.eReturnCode = CODE_UNUSED;
	 
	memcpy(psMessage->szMessage, szMessage, MESSAGE_LENGTH);
	psMessage->sHeader.nMessageLength = sizeof(SResponseMessageChat);
	SDLNet_Write16(nSender, &psMessage->nSender);
	SDLNet_Write16(psMessage->sHeader.nMessageLength, &psMessage->sHeader.nMessageLength);
	//cout << "mess length hier: " << psMessage->sHeader.nMessageLength << endl;
	AddMessage(pChatMessage);
}

SPlayerInfo* CServerClient::GetInfo()
{
	return &m_sPlayerInfo;
}

void CServerClient::CorrectIncomingByteOrder(SGamePlayerStats* pBuffer, u8 nAmount)
{
	// Nur float Werte - Keine konvertierung benötigt
}

void CServerClient::CorrectOutgoingByteOrder(SGamePlayerStats* pBuffer, u8 nAmount)
{
	// Nur float Werte - Keine konvertierung benötigt
}

void CServerClient::CorrectOutgoingByteOrder(SPlayerInfo* pBuffer, u8 nAmount)
{
	for (u8 i = 0; i < nAmount; i++)
	{
		SDLNet_Write16(pBuffer[i].nID, &pBuffer[i].nID);
	}
}

void CServerClient::CorrectOutgoingByteOrder(SGameInfo* pBuffer, u8 nAmount)
{
	for (u8 i = 0; i < nAmount; i++)
	{
		SDLNet_Write16(pBuffer[i].nID, &pBuffer[i].nID);
	}
}
