#ifndef SERVERCLIENT_HPP_
#define SERVERCLIENT_HPP_

#include "../globals.hpp"
#include "../tools/BaseThread.hpp"
#include "../net/SocketFramework.hpp"
#include "../net/NetworkMessages.hpp"
#include "../net/Message.hpp"
#include <vector>
#include <iostream>
#include <ctime>

class CServer;
class CServerGame;

///Pro Spieler erstellten Client mittels Thread zum Verwalten der
///Verbindungsdaten, sowie der Authentifizierung

class CServerClient : public BaseThread
{

public:
	/// Konstruktor
	/// @param	psSocket	TCPsocket des Clients
	/// @param	pServer		Zeiger auf den Spieleverwatungsserver

	CServerClient(TCPsocket psSocket,CServer* pServer);

	/// Destruktor

	~CServerClient();

	/// Message an die Schlange fuer den Client anhaengen
	/// @param psMessage Zeiger auf die Nachrichtenklasse

	void AddMessage(CMessage* psMessage);

	/// Endlosschleife in der die Kommunikation mit dem Client abgehandelt wird
	int Run();

	/// Nachricht aufbereiten zum Abspeichern in der Messageschlange
	/// @param nSender ID des sendenden Spielers
	/// @param szMessage Zeiger auf die Nachricht an den Client
	void Chat(u16 nSender, char* szMessage);
	
	// Gibt die ID und Namen des Spielers zurueck
	SPlayerInfo* GetInfo();
	
	/// Erzeugt eine GAME_GET_MAP Antwortnachricht mit Kartendaten
	/// @param	nMapID	ID der karte mit der sie vom Server abgerufen werden kann
	bool SendMap(u8 nMapID);
	
	/// Sendet eine GAME_END_LEVEL Antwortnachricht an alle Clients
	/// um zu informieren das ein neues Level geladen werden muss
	void ProceedLevel(u8 nPlayerStartPos, u32 nStartTime);
	
private:
	/// Client-Zähler, wird auch benutzt um Auth-IDs zu erzeugen
	static u16 nClientCounter;
	
	/// Semaphore zum Thread-sicheren Zugriff auf Daten
	SDL_mutex* m_pMutex;

	///Authentifizierungs-Token f�r den Spieler
	u32 m_nAuthToken;

	///Server-Objekt
	CServer* m_pServer;

	///Zum Client gehoerender Socket
	TCPsocket m_sSocket;
	
	//ID und Name des Players
	SPlayerInfo m_sPlayerInfo;
	
	//Nachrichtenstruct
	std::vector<CMessage*> m_vMessageQueue;
	
	/// True wenn Client gerade an einem Spiel beteilig ist, sonst false
	bool m_bInGame;
	
	/// Das Spiels an dem der Client momentan beteiligt ist.
	CServerGame* m_pCurrentGame;
	
	/// Byte-Order von eingehende Daten eines SGamePlayerStats
	/// wiederherstellen
	/// @param	pBuffer	Zeiger auf Daten
	/// @param	nAmount	Anzahl Speicherstrukturen die zu korregieren sind

	static void CorrectIncomingByteOrder(SGamePlayerStats* pBuffer, u8 nAmount);
	
	/// Byte-Order von ausgehenden Daten eines SGamePlayerStats
	/// Netzwerkfähig machen
	/// @param	pBuffer	Zeiger auf Daten
	/// @param	nAmount	Anzahl Speicherstrukturen die zu korregieren sind

	static void CorrectOutgoingByteOrder(SGamePlayerStats* pBuffer, u8 nAmount);
	
	/// Byte-Order von ausgehenden Daten eines SPlayerInfo
	/// Netzwerkfähig machen
	/// @param	pBuffer	Zeiger auf Daten
	/// @param	nAmount	Anzahl Speicherstrukturen die zu korregieren sind

	static void CorrectOutgoingByteOrder(SPlayerInfo* pBuffer, u8 nAmount);
	
	/// Byte-Order von ausgehenden Daten eines SGameInfo
	/// Netzwerkfähig machen
	/// @param	pBuffer	Zeiger auf Daten
	/// @param	nAmount	Anzahl Speicherstrukturen die zu korregieren sind

	static void CorrectOutgoingByteOrder(SGameInfo* pBuffer, u8 nAmount);
	
	/// Empf�ngt nNumMessages Nachrichten und l�st die erforderlichen Aktionen aus
	/// @param	nNumMessages	Anzahl zu empfangender Nachrichten
	/// @return	true wenn alle Nachrichten empfangen werden konnten sonst false

	bool HandleMessages(u8 nNumMessages, bool bClientNotAuthenticated);
	
	/// Benutzt zur Auswertung von CLIENT_LOGIN-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleClientLogin(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von CLIENT_LOGOUT-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleClientLogout(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von CLIENT_CHAT-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleClientChat(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von CLIENT_SYNC-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleClientSync(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von SERVER_CREATE_GAME-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleServerCreateGame(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von SERVER_COUNT_GAMES-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleServerCountGames(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von SERVER_GET_GAMES-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleServerGetGames(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von SERVER_GET_GAME-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleServerGetGame(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von SERVER_GET_PLAYER-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleServerGetPlayer(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von SERVER_GET_MAPS-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleServerGetMaps(CMessage* pMessage);
	
	/// Empfaengt eine neue Karte
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameGetMap(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_MAP_CONFIG-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameMapConfig(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_START-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameStart(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_JOIN-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameJoin(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_LEAVE-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameLeave(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_PLAYER_DIE-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGamePlayerDie(CMessage* pMessage);

	/// Benutzt zur Auswertung von GAME_PUSH_PROGRESS-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGamePushProgress(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_GET_PLAYER-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameGetPlayer(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_GET_PLAYERINFO-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameGetPlayerInfo(CMessage* pMessage);
	
	/// Benutzt zur Auswertung von GAME_NEXT_LEVEL-Nachrichten
	/// @param	pMessage	Die eingehende Nachricht (Header)
	/// @return	Ein boolean der den Erfolg der Auswertung anzeigt

	bool HandleGameNextLevel(CMessage* pMessage);
	
	/// Erzeugt eine GAME_PROGRESS-Nachricht um dem Client
	/// den aktuellen Spielverlauf zu schicken

	void GenerateGameProgressMessage();
	
	/// Sendet alle Nachrichten die in m_vMessageQueue enthalten sind
	/// @return	true wenn alle Daten gesendet werden konnten sonst false

	bool SendMessageQueue();
};

#endif ///SERVERCLIENT_HPP_///

