#include "ServerGame.hpp"
#include <ctime>

u16 CServerGame::m_nGameCounter = LAN_GAME_ID;

CServerGame::CServerGame(char* szName, u8 nMaxPlayer, bool bAllowLateJoin, bool bAllowVisitors, char* szPassword)
{
	m_sGameInfo.nID = m_nGameCounter;
	/// alle Zeichen des Strings auf 0 setzen
	memset(m_sGameInfo.szName, 0, GAME_NAME_LENGTH);
	memcpy(m_sGameInfo.szName, szName, GAME_NAME_LENGTH);
	
	memset(m_szPassword, 0, GAME_PASSWD_LENGTH);
	if (szPassword == NULL || strlen(szPassword) < 1)
	{
		m_sGameInfo.bUsePassword = false;
	}
	else
	{
		m_sGameInfo.bUsePassword = true;
		memcpy(m_szPassword, szPassword, GAME_PASSWD_LENGTH);
	}
	
	m_sGameInfo.eState = SERVERGAME_CONFIG;
	m_sGameInfo.nNumPlayers = 0;
	m_sGameInfo.nNumClients = 0;
	m_sGameInfo.nMaxPlayers = nMaxPlayer;
	m_sGameInfo.bLateJoin = bAllowLateJoin;
	m_sGameInfo.bAllowVisitors = bAllowVisitors;
	
	m_sGameInfo.nNumLevels = 0;
	m_sGameInfo.nCurrentLevel = 0;
	
	m_pMutex = SDL_CreateMutex();
	
	m_nGameCounter++;
}

SGameInfo* CServerGame::GetGameInfo()
{
	return &m_sGameInfo;
}

bool CServerGame::Join(CServerClient* pPlayer, char* szPassword)
{
	SGamePlayer sPlayerInfo;
	
	SDL_LockMutex(m_pMutex);
	
	// Beitritt nach Spielbeginn nur erlaubt wenn bLateJoin true ist
	if (m_sGameInfo.nNumPlayers == m_sGameInfo.nMaxPlayers || 
			(m_sGameInfo.eState != SERVERGAME_CONFIG && m_sGameInfo.bLateJoin == false))
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	// Passwortprüfung
	if (m_sGameInfo.bUsePassword == true && (szPassword == NULL || strncmp(m_szPassword, szPassword, GAME_PASSWD_LENGTH) != 0))
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	sPlayerInfo.pConnection = pPlayer;
	memset(&sPlayerInfo.sStats, 0, sizeof(SGamePlayerStats));
	sPlayerInfo.sInfo.nRoundsWon = 0;
	sPlayerInfo.sInfo.eState = PLAYER_JOINED;
	sPlayerInfo.sInfo.eRole = (m_mPlayer.size() == 0) ? ADMIN : PARTICIPANT;
	
	/// Spieler registrieren
	m_mPlayer[pPlayer->GetInfo()->nID] = sPlayerInfo;
	m_sGameInfo.nNumPlayers++;
	m_sGameInfo.nNumClients++;
	SDL_UnlockMutex(m_pMutex);
	
	return true;
}

bool CServerGame::Visit(CServerClient* pPlayer)
{
	SGamePlayer sPlayerInfo;
		
	SDL_LockMutex(m_pMutex);
	if (m_sGameInfo.bAllowVisitors == false)
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	sPlayerInfo.pConnection = pPlayer;
	sPlayerInfo.sInfo.eRole = SPECTATOR;
	
	m_mPlayer[pPlayer->GetInfo()->nID] = sPlayerInfo;
	m_sGameInfo.nNumClients++;
	SDL_UnlockMutex(m_pMutex);
		
	return true;
}

void CServerGame::Leave(u16 nPlayer)
{
	SDL_LockMutex(m_pMutex);
	
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
	
	if (iter == m_mPlayer.end())
	{
		SDL_UnlockMutex(m_pMutex);
		return;
	}
	
	m_sGameInfo.nNumClients--;
	if (iter->second.sInfo.eRole != SPECTATOR)
	{
		m_sGameInfo.nNumPlayers--;
	}
	
	m_mPlayer.erase(iter);
	SDL_UnlockMutex(m_pMutex);
}

void CServerGame::Delete()
{
	// TODO: Nutzer�berpr�fung? Nutzer-ID als Parameter und auf ADMIN pr�fen
}

u8 CServerGame::GetPlayerState(u16 nPlayer)
{
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
	
	if (iter == m_mPlayer.end())
	{
		return PLAYER_DONT_EXISTS;
	}
	
	return iter->second.sInfo.eState;
}

SGamePlayer* CServerGame::GetSpecificPlayer(u16 nPlayer)
{
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
	
	if (iter == m_mPlayer.end())
	{
		return NULL;
	}
	
	return &iter->second;
}

void CServerGame::Chat(CServerClient* pSender, char* szMessage)
{
	std::map<u16, SGamePlayer>::iterator iter;
	for (iter = m_mPlayer.begin(); iter != m_mPlayer.end(); iter++)
	{
		iter->second.pConnection->Chat(pSender->GetInfo()->nID, szMessage);
	}
}

bool CServerGame::AddMap(u16 nPlayer, u8 nMapID)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
	
	if (iter == m_mPlayer.end() || iter->second.sInfo.eRole != ADMIN)
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	m_vLevels.push_back(nMapID);
	m_sGameInfo.nNumLevels++;
	SDL_UnlockMutex(m_pMutex);
	return true;
}

bool CServerGame::RemoveMap(u16 nPlayer, u8 nPosition)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
	
	if (iter == m_mPlayer.end() || iter->second.sInfo.eRole != ADMIN || 
			nPosition >= m_vLevels.size())
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}

	std::vector<u8>::iterator map = m_vLevels.begin() + nPosition;
	m_vLevels.erase(map);
	m_sGameInfo.nNumLevels--;
	SDL_UnlockMutex(m_pMutex);
	return true;
}

bool CServerGame::GetMapIDByLevel(u8 nLevel, u8* pnMapID)
{
	if (nLevel >= m_vLevels.size())
	{
		return false;
	}
	else
	{
		*pnMapID = m_vLevels[nLevel];
		return true;
	}
}

bool CServerGame::Start(u16 nPlayer)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
		
	if (iter == m_mPlayer.end() || m_vLevels.size() == 0)
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	iter->second.sInfo.eState = PLAYER_READY;
	
	/// If all players has called the start method the game state will change to GAME_RUNNING
	m_sGameInfo.eState = SERVERGAME_RUNNING;
	for (iter = m_mPlayer.begin(); iter != m_mPlayer.end(); iter++)
	{
		if (iter->second.sInfo.eRole != SPECTATOR && iter->second.sInfo.eState != PLAYER_READY)
		{
			m_sGameInfo.eState = SERVERGAME_CONFIG;
			break;
		}
	}

	if (m_sGameInfo.eState == SERVERGAME_RUNNING)
	{
		SDL_UnlockMutex(m_pMutex);
		DEBUG_PRINTL("Game is now RUNNING!");
		m_sGameInfo.nCurrentLevel = -1;
		NextLevel();	//ruft LockMutex() auf!
		SDL_LockMutex(m_pMutex);
	}

	SDL_UnlockMutex(m_pMutex);
	return true;
}

bool CServerGame::Die(u16 nPlayer)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
		
	if (iter == m_mPlayer.end() || m_vLevels.size() == 0)
	{
		SDL_UnlockMutex(m_pMutex);
		return false;
	}
	
	iter->second.sInfo.eState = PLAYER_INGAME_DEAD;
	
	/// If all players are dead, the game must proceed to the next level
	bool allDead = true;
	for (iter = m_mPlayer.begin(); iter != m_mPlayer.end(); iter++)
	{
		if (iter->second.sInfo.eRole != SPECTATOR && iter->second.sInfo.eState != PLAYER_INGAME_DEAD)
		{
			allDead = false;
			break;
		}
	}

	if (allDead == true)
	{
		SDL_UnlockMutex(m_pMutex);
		DEBUG_PRINTL("Everybody is dead - go to next level!");
		NextLevel();
		SDL_LockMutex(m_pMutex);
	}

	SDL_UnlockMutex(m_pMutex);
	return true;
}

void CServerGame::NextLevel()
{
	if (m_sGameInfo.eState != SERVERGAME_RUNNING)
	{
		return;
	}

	DEBUG_PRINTL("Changed Level - sending Level-Data #" << m_sGameInfo.nCurrentLevel << " to everybody");

	m_sGameInfo.nCurrentLevel++;
	if (m_sGameInfo.nCurrentLevel >= m_sGameInfo.nNumLevels)
	{
		m_sGameInfo.eState = SERVERGAME_END;
	}

	std::map<u16, SGamePlayer>::iterator iter;

	SDL_LockMutex(m_pMutex);
	if (m_sGameInfo.eState != SERVERGAME_END)
	{
		for (iter = m_mPlayer.begin(); iter != m_mPlayer.end(); iter++)
		{
			// Revive each player
			iter->second.sInfo.eState = PLAYER_INGAME;
			iter->second.pConnection->SendMap(m_vLevels[m_sGameInfo.nCurrentLevel]);
		}
	}

	u8 nPos = 0;
	u32 nStartTime = time(NULL) + GAME_NEXT_LEVEL_PAUSE_SEC;
	for (iter = m_mPlayer.begin(); iter != m_mPlayer.end(); iter++)
	{
		iter->second.pConnection->ProceedLevel(nPos++, nStartTime);
	}
	SDL_UnlockMutex(m_pMutex);
}

void CServerGame::GetPlayerStats(u16* pnOrder, SGamePlayerStats* psPlayerStats)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.begin();
	
	for (int i = 0; i < m_sGameInfo.nNumPlayers; i++)
	{
		// Status von Zuschauern ist uninterssant
		while (iter->second.sInfo.eRole == SPECTATOR)
		{
			iter++;
		}
		
		SDLNet_Write16(iter->first, &pnOrder[i]); //musste hier rein schon
		memcpy(psPlayerStats + i, &iter->second.sStats, sizeof(SGamePlayerStats));
		iter++;
	}
	SDL_UnlockMutex(m_pMutex);
}

void CServerGame::SetPlayerStats(u16 nPlayer, SGamePlayerStats* psPlayerStats)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.find(nPlayer);
		
	if (iter == m_mPlayer.end())
	{
		SDL_UnlockMutex(m_pMutex);
		return;
	}
	
	memcpy(&iter->second.sStats, psPlayerStats, sizeof(SGamePlayerStats));
	SDL_UnlockMutex(m_pMutex);
}

void CServerGame::GetPlayerInfo(u16* pnOrder, SGamePlayerInfo* psPlayerInfos)
{
	SDL_LockMutex(m_pMutex);
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.begin();
		
	for (int i = 0; i < m_sGameInfo.nNumPlayers; i++)
	{
		// Status von Zuschauern ist uninterssant
		while (iter->second.sInfo.eRole == SPECTATOR)
		{
			iter++;
		}
		
		SDLNet_Write16(iter->first, &pnOrder[i]);	//musste hier rein schon
		memcpy(psPlayerInfos + i, &iter->second.sInfo, sizeof(SGamePlayerInfo));
		iter++;
	}
	SDL_UnlockMutex(m_pMutex);
}

void CServerGame::GetPlayer(SPlayerInfo* pPlayerInfos)
{
	SDL_LockMutex(m_pMutex);
	CServerClient* pPlayer;
	std::map<u16, SGamePlayer>::iterator iter = m_mPlayer.begin();
	
	for (u16 i = 0; i < m_sGameInfo.nNumPlayers; i++)
	{
		// Status von Zuschauern ist uninterssant
		while (iter->second.sInfo.eRole == SPECTATOR)
		{
			iter++;
		}
		
		pPlayer = iter->second.pConnection;
		memcpy(pPlayerInfos + i, pPlayer->GetInfo(), sizeof(SPlayerInfo));
		iter++;
	}
	SDL_UnlockMutex(m_pMutex);
}
