#ifndef SERVERGAME_HPP_
#define SERVERGAME_HPP_

#include <vector>
#include <map>
#include "../net/NetworkTypes.hpp"
#include "GameManager.hpp"
#include "ServerClient.hpp"

struct SGamePlayer
{
	CServerClient* pConnection;
	SGamePlayerStats sStats;
	SGamePlayerInfo sInfo;
};

/// Spielverwaltung

class CServerGame
{
	
public:
	/// Konstruktor
	/// @param	szName	Spielname
	/// @param	nMaxPlayer	Maximale Spieleranzahl
	/// @param	bAllowLateJoin	Erlaube das beitreten nachdem das Spiel gestartet wurde
	/// @param	bAllowVisitors	Erlaube Zuschauer
	/// @param	szPassword	Zugangspasswort

	CServerGame(char* szName, u8 nMaxPlayer, bool bAllowLateJoin, bool bAllowVisitors, char* szPassword);
	
	/// Gibt Spieleinformationen zur�ck
	/// @return	Spieleinformationen

	SGameInfo* GetGameInfo();
	
	/// Dem Spiel beitreten
	/// Der erste Spieler der beitritt wird zum Admin des Spiels
	/// @param	pPlayer	Eine Instanz der Client-Kommunikation
	/// @param	szPassword	Passwort zum beitritt
	/// @return	true wenn beigetreten werden konnte sonst false

	bool Join(CServerClient* pPlayer, char* szPassword);
	
	/// Als Zuschauer registrieren
	/// @param	pPlayer	Eine Instanz der Client-Kommunikation
	/// @return	true wenn zugeschaut werden kann sonst false

	bool Visit(CServerClient* pPlayer);
	
	/// Verlassen des Spiels
	/// @param	nPlayer Zuschauer oder Mitspieler der das Spiel verl�sst

	void Leave(u16 nPlayer);
	
	/// L�scht ein Spiel - Alle Spieler und Zuschauer werden vorher entfernt

	void Delete();
	
	/// Gibt den momentanen Status eines Spielers zur�ck
	/// @param	nPlayer	Spieler-ID
	/// @return	Der momentane Status

	u8 GetPlayerState(u16 nPlayer);
	
	/// Sendet eine Nachricht an alle Mitspieler (nicht an Zuschauer)
	/// @param	pSender	referenz auf sendenden client
	/// @param	szMessage	Die zu sendende Nachricht

	void Chat(CServerClient* pSender, char* szMessage);
	
	/// F�gt eine Level zu den zu Spielenden hinzu
	/// @param	nPlayer	Aufrufender Spieler (nur Spiel-Admin kann Level hinzuf�gen)
	/// @param	nMapID	ID des Levels
	/// @return	true wenn Level hinzugef�gt werden konnte sonst false

	bool AddMap(u16 nPlayer, u8 nMapID);
	
	/// Entfernt ein Level aus den zu Spielenden. (Es wird immer das Erste gefundene Level entfernt sollten mehrere vorhanden sein)
	/// @param	nPlayer	Aufrufender Spieler (nur Spiel-Admin kann Level entfernen)
	/// @param	nPosition	Nummer des Levels in der Reihenfolge der zu spielenden Level
	/// @return	true wenn Level entfernt werden konnte sonst false

	bool RemoveMap(u16 nPlayer, u8 nPosition);
	
	/// Gibt die Karten-ID zu einem Level zurueck
	/// @param	nLevel	Level
	/// @param	pnMapID	Zeiger an dessen Stelle die Map-ID geschrieben wird
	/// @return	true wenn eine Karten-ID gefunden wurde sonst false

	bool GetMapIDByLevel(u8 nLevel, u8* pnMapID);
	
	/// Signalisiert Spielbereitschaft eines Mitspielers
	/// @param	nPlayer	Aufrufender Spieler
	/// @return	true wenn das Spiel gestartet werden konnte oder false wenn
	/// 			entweder der Spieler nicht teilnimmt am Spiel oder noch keine
	/// 			Karten dem Spiel hinzugefuegt wurden

	bool Start(u16 nPlayer);
	
	/// Zeigt an, dass ein Spieler tot ist
	bool Die(u16 nPlayer);
	
	/// Aufgerufen wenn ein Spieler das Ziel erreicht
	/// Schickt automatisch neue Karteninfos an die Spieler

	void NextLevel();
	
	/// Schreibt Spielerinformationen (Momentaner Spielstand) in einem Puffer
	/// @param	pnOrder	Ein Puffer in den die ID der Spieler in der Reihenfolge
	/// 					kopiert werden die mit der Reihenfolge der Spielinformationen
	/// 					in psPlayerStats �bereinstimmt
	/// @param	psPlayerStats	Puffer in den Spielinformationen geschrieben werden

	void GetPlayerStats(u16* pnOrder, SGamePlayerStats* psPlayerStats);
	
	/// Aktualisiert Spielerinformationen eines Spielers
	/// @param	nPlayer	Aufrufender Spieler
	/// @param	psPlayerStats	Momentaner Spielstand des Spielers

	void SetPlayerStats(u16 nPlayer, SGamePlayerStats* psPlayerStats);
	
	/// Schreibt Spielerinformationen (Anzahl gewonnener Runden, Rolle im Spiel, Momentaner Status) in einen Puffer
	/// @param	pnOrder	Ein Puffer in den die ID der Spieler in der Reihenfolge
	/// 					kopiert werden die mit der Reihenfolge der Spielinformationen
	/// 					in psPlayerInfos �bereinstimmt
	/// @param	psPlayerInfos	Puffer in den Spielinformationen geschrieben werden

	void GetPlayerInfo(u16* pnOrder, SGamePlayerInfo* psPlayerInfos);
	
	/// Schreibt Spielerinformationen (SPlayerInfo) jedes Spielers
	/// in einen Puffer
	/// @param	pPlayerInfos	Der Zielpuffer

	void GetPlayer(SPlayerInfo* pPlayerInfos);

	/// Gibt Zeiger auf einen bestimmten Spieler zurueck
	/// @param	nPlayer	Spieler-ID

	SGamePlayer* GetSpecificPlayer(u16 nPlayer);
	
private:
	/// Ausgew�hlte Level
	std::vector<u8> m_vLevels;
	
	/// Spieleinfomationen
	SGameInfo m_sGameInfo;
	
	/// Zuordnung Spieler-ID zu Spielerinformationen
	std::map<u16, SGamePlayer> m_mPlayer;
	
	/// Semaphore
	SDL_mutex* m_pMutex;
	
	/// Passwort f�r Zugangskontrolle
	char m_szPassword[GAME_PASSWD_LENGTH];
	
	/// Z�hler um eindeutige Spiele-IDs zu erzeugen
	static u16 m_nGameCounter;
};

#endif
