#ifndef CVEC2D_HPP
#define CVEC2D_HPP

#include <math.h>
#include <iostream>
using namespace std;

#ifndef MIN
	#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#endif

#ifndef MAX
	#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))
#endif

///	Vektor im 2D-Raum
///	@author amir@cfxweb.net,
///			David Scherfgen,
///			Markus Konrad
/// @see CVec3D
class CVec2D
{
	public:
		float x;
		float y;

		/// Standardkonstruktor: Als Nullvektor initialisieren.
		CVec2D() { CVec2D(0.0f); }

		/// Initialisierung mit 3 Floats
		/// @param fX X-Koord.
		/// @param fY Y-Koord.
		CVec2D(const float fX, const float fY) { x = fX; y = fY; }

		/// Initialisierung mit Float-Array
		/// @param f Float-Array mit x, y Koord.
		CVec2D(const float f[2]) { x = f[0]; y = f[1]; }

		/// Initialisierung mit CVec2D
		/// @param v CVec2D Objekt
		CVec2D(const CVec2D& v) { x = v.x; y = v.y; }

		/// Initialisierung mit einem Floatwert
		/// @param f Alle Koord. erhalten diesen Wert
		CVec2D(const float f) { x = f;	y = f; }

		CVec2D operator+(const CVec2D& v) { return CVec2D(x + v.x, y + v.y); }
		CVec2D operator-(const CVec2D& v) { return CVec2D(x - v.x, y - v.y); }
		CVec2D operator*(const CVec2D& v) { return CVec2D(x * v.x, y * v.y); }
		CVec2D operator/(const CVec2D& v) { return CVec2D(x / v.x, y / v.y); }

		CVec2D operator*(const float f) { return CVec2D(x * f, y * f); }
		CVec2D operator/(const float f) { return CVec2D(x / f, y / f); }

		/// Vektorrichtung umkehren
		CVec2D operator-(void) { return CVec2D(-x, -y); }

		CVec2D& operator+=(const CVec2D& v) {	*this = *this + v; return *this; }
		CVec2D& operator-=(const CVec2D& v) {	*this = *this - v; return *this; }
		CVec2D& operator*=(const CVec2D& v) {	*this = *this * v; return *this; }
		CVec2D& operator/=(const CVec2D& v) {	*this = *this / v; return *this; }
		CVec2D& operator*=(const float f) { *this = *this * f; return *this; }
		CVec2D& operator/=(const float f) { *this = *this / f; return *this; }

		/// Betrag dieses Vektors ohne Wurzel
		/// @return quadratischer Betrag
		inline float SqrLength(void) { return (x * x + y * y); }

		/// Betrag dieses Vektors (entspr. L?nge)
		/// @return Betrag
		inline float Length(void) {	return (float)sqrt(SqrLength()); }

		/// Skalarprodukt dieses Vektors mit einem anderen Vektor berechnen
		/// @return Skalarprodukt
		inline float ScalarProduct(const CVec2D& v) {	return ( x * v.x + y * v.y ); }

		/// Skalarprodukt zweier Vektoren berechnen
		/// @return Skalarprodukt
		inline static float ScalarProduct(const CVec2D& a, const CVec2D& b) { return a.x * b.x + a.y * b.y; }

		/// Determinante zweier Vektoren berechnen
		/// @return Determinante
		inline static float Determinant(const CVec2D& a, const CVec2D& b) { return a.x * b.y - a.y * b.x; }

		/// Normalisieren eines Vektors
		/// D.h. der Betrag d. Vektors = 1
		inline void Normalize(void);

		/// Schnittwinkel zwischen diesem Vektor und einem Vektor v berechnen
		/// @param v Vektor v
		/// @return Schnittwinkel
		inline float IntersectAngle(CVec2D& v) {	float f = ( ScalarProduct(v) / Length()) * (1.0f / v.Length() ); return acos(f); }	//Schnittwinkel dieses Vektors mit einem anderen Vektor berechnen

		/// Rotation dieses Vektors um einen Winkel in Radianten
		/// @param fRadAngle Rotationswinkel in Radianten
		inline void Rotate(float fRadAngle);
/*
		/// Test ob ein rotiertes Rechteck ein anderes rotiertes Rechteck trifft
		/// @author Oren Becker, 2001
		/// @see http://www.ragestorm.net/tutorial?id=22
		static bool RotRectHitsRotRect(const CVec2D& vRect1Center, const CVec2D& vRect1Shape, const float fRec1Angle,
										const CVec2D& vRect2Center, const CVec2D& vRect2Shape, const float fRec2Angle);
*/
		/// Statische Funktion zum Test ob ein Liniesegment ein anderes Liniensegment schneidet im 2D-Raum
		inline static bool LineHitsLine(
					CVec2D& vLineAStart,
                    CVec2D& vLineAEnd,
                    CVec2D& vLineBStart,
                    CVec2D& vLineBEnd,
                    CVec2D* pvIntersection
		)
		{
			//Richtungsvektoren bestimmen
			CVec2D vLineADir(vLineAEnd - vLineAStart);
			CVec2D vLineBDir(vLineBEnd - vLineBStart);

			//s f?r den Schnittpunkt berechnen
			const float fDet1 = CVec2D::Determinant(vLineBStart - vLineAStart, -vLineBDir);
			const float fDet2 = CVec2D::Determinant(vLineADir, -vLineBDir);

			if (fDet1 == 0.0 && fDet2 == 0.0)	//gleich
			{
				*pvIntersection = vLineAStart;
				return true;
			}
			else if (fDet2 == 0.0)	//parallel
				return false;

			const float s = fDet1 / fDet2;
			
			if (s < 0.0 || s > 1.0)	//ausserhalb der Geraden
				return false;

			//Schnittpunkt berechnen
			*pvIntersection = vLineAStart + vLineADir * s;
			
			//Testen ob der Schnittpunkt wirklich im gefragten Liniensegment ist
			if (pvIntersection->x > MAX(vLineBStart.x, vLineBEnd.x))
				return false;
			if (pvIntersection->y > MAX(vLineBStart.y, vLineBEnd.y))
				return false;
			if (pvIntersection->x < MIN(vLineBStart.x, vLineBEnd.x))
				return false;
			if (pvIntersection->y < MIN(vLineBStart.y, vLineBEnd.y))
				return false;

			return true;
		}

		/// Statische Funktion zum Test ob eine Linie einen Kreis schneidet im 2D-Raum
		/// @author David Scherfgen
		inline static bool LineHitsCircle(
					CVec2D& vLineStart,
                    CVec2D& vLineEnd,
                    CVec2D& vCircleCenter,
                    const float fRadius,
                    CVec2D* pvIntersection = 0
		)
		{
			// r? vorberechnen
			const float fRadSq = fRadius * fRadius;

			// (p - m) vorberechnen
			CVec2D vPMinusM(vLineStart - vCircleCenter);

			// Wenn der Startpunkt der Linie schon im Kreis liegt,
			// dann brechen wir sofort ab.
			if(vPMinusM.SqrLength() <= fRadSq)
			{
				// Als Schnittpunkt nehmen wir einfach
				// den Startpunkt der Linie.
				if (pvIntersection) *pvIntersection = vLineStart;
				return true;
			}

			// Richtungsvektor der Linie berechnen (u)
			CVec2D vLineDir(vLineEnd - vLineStart);

			// u * (p - m) vorberechnen
			const float fUDotPMinusM = CVec2D::ScalarProduct(vLineDir, vPMinusM);

			// u? vorberechnen
			float fLineDirSq = vLineDir.SqrLength();

			// Die Diskriminante berechnen:
			//   (u * (p - m))?
			// - (u? * ((p - m)? - r?))
			const float fD = fUDotPMinusM * fUDotPMinusM
							- fLineDirSq * (vPMinusM.SqrLength() - fRadSq);

			// Ist die Diskriminante negativ, null oder positiv?
			if(fD < 0.0f) return false;
			else if(fD < 0.0001f)
			{
				// Die Diskriminante ist (ungef?hr) null.
				// Die gesamte Wurzel entf?llt und die L?sung ist:
				// (-u * (p - m)) / u?.
				// Wir m?ssen nur noch pr?fen, ob der Wert
				// im korrekten Bereich [0; 1] liegt.
				const float fS = -fUDotPMinusM / fLineDirSq;
				if(fS < 0.0f || fS > 1.0f) return false;
				else
				{
					// Ber?hrpunkt!
					if (pvIntersection) *pvIntersection = vLineStart + vLineDir * fS;
					return true;
				}
			}
			else
			{
				// Die Gerade schneidet den Kreis zweimal.
				// Uns interessiert nur die kleinere L?sung f?r s,
				// denn das ist die Stelle, wo die Linie den Kreis
				// "zum ersten Mal" schneidet.
				// Diese L?sung berechnen wir nun.
				const float fS = (-fUDotPMinusM - sqrtf(fD)) / fLineDirSq;
				if(fS < 0.0f || fS > 1.0f) return false;
				else
				{
				   if (pvIntersection) *pvIntersection = vLineStart + vLineDir * fS;
					return true;
				}
			}
		}
};

inline void CVec2D::Rotate(float fRadAngle)
{
	//Rotation von Punkten im 2-Dim. Raum:
	//x' = x * cos (a) - y * sin(a)
	//y' = y * cos (a) + x * sin(a)

	float fCosVal	= cosf(fRadAngle);
	float fSinVal	= sinf(fRadAngle);
	float fNewX, fNewY;

	fNewX = x * fCosVal - y * fSinVal;
	fNewY = y * fCosVal + x * fSinVal;

	x = fNewX;
	y = fNewY;
}

inline void CVec2D::Normalize(void)
{
	float fLen = Length();
	if (!fLen) return;
	fLen = 1.0f / fLen;

	x *= fLen;
	y *= fLen;
}

#endif
