#ifndef CVEC3D_HPP
#define CVEC3D_HPP

#include <math.h>

///	Vektor im 3D-Raum
///	@author amir@cfxweb.net,
///			Erweitert von Markus Konrad
/// @see http://www.gamespp.com/algorithms/collisionDetectionTutorial01.html
class CVec3D
{
	public:
		float x;
		float y;
		float z;

		/// Standardkonstruktor: Als Nullvektor initialisieren.
		CVec3D() { CVec3D(0.0f); }

		/// Initialisierung mit 3 Floats
		/// @param fX X-Koord.
		/// @param fY Y-Koord.
		/// @param fZ Z-Koord.
		CVec3D(const float fX,const float fY,const float fZ) { x = fX; y = fY; z = fZ; }

		/// Initialisierung mit Float-Array
		/// @param f Float-Array mit x, y, z Koord.
		CVec3D(const float f[3]) { x = f[0]; y = f[1]; z = f[2]; }

		/// Initialisierung mit einem Floatwert
		/// @param f Alle Koord. erhalten diesen Wert
		CVec3D(const float f) { x = f;	y = f; z = f; }

		CVec3D operator+(const CVec3D& v) { return CVec3D(x + v.x, y + v.y, z + v.z); }
		CVec3D operator-(const CVec3D& v) { return CVec3D(x - v.x, y - v.y, z - v.z); }
		CVec3D operator*(const CVec3D& v) { return CVec3D(x * v.x, y * v.y, z * v.z); }
		CVec3D operator/(const CVec3D& v) { return CVec3D(x / v.x, y / v.y, z / v.z); }

		CVec3D operator*(const float f) { return CVec3D(x * f, y * f, z * f); }
		CVec3D operator/(const float f) { return CVec3D(x / f, y / f, z / f); }

		/// Vektorrichtung umkehren
		CVec3D operator-(void) { return CVec3D(-x, -y, -z); }

		CVec3D& operator+=(const CVec3D& v) {	*this = *this + v; return *this; }
		CVec3D& operator-=(const CVec3D& v) {	*this = *this - v; return *this; }
		CVec3D& operator*=(const CVec3D& v) {	*this = *this * v; return *this; }
		CVec3D& operator/=(const CVec3D& v) {	*this = *this / v; return *this; }
		CVec3D& operator*=(const float f) { *this = *this * f; return *this; }
		CVec3D& operator/=(const float f) { *this = *this / f; return *this; }

		/// Betrag dieses Vektors ohne Wurzel
		/// @return quadratischer Betrag
		inline float SqrLength(void) { return (x * x + y * y + z * z); }

		/// Betrag dieses Vektors (entspr. Lnge)
		/// @return Betrag
		inline float Length(void) {	return (float)sqrt(SqrLength()); }

		/// Skalarprodukt dieses Vektors mit einem anderen Vektor berechnen
		/// @return Skalarprodukt
		inline float ScalarProduct(const CVec3D& v) {	return ( x * v.x + y * v.y + z * v.z ); }

		/// Skalarprodukt zweier Vektoren berechnen
		/// @return Skalarprodukt
		inline static float ScalarProduct(const CVec3D& a, const CVec3D& b) { return a.x * b.x + a.y * b.y + a.z * b.z; }

		/// Kreuzprodukt zweier Vektoren bilden und als neuen Vektor zurckgeben
		/// @param a Vektor a
		/// @param b Vektor b
		/// @return Kreuzprodukt a x b
		inline static CVec3D CrossProduct(const CVec3D& a, const CVec3D& b)
		{
			CVec3D c;

			c.x = ( a.y * b.z ) - ( a.z * b.y );
			c.y = ( a.z * b.x ) - ( a.x * b.z );
			c.z = ( a.x * b.y ) - ( a.y * b.x );

			return c;
		}

		/// Kreuzprodukt dieses Vektors mit v berechnen
		/// @param v Vektor v
		/// @return Kreuzprodukt dieses Vektors mit v
		inline CVec3D CrossProduct(const CVec3D& v) { return CrossProduct(*this, v); }

		inline void Normalize(void);

		/// Schnittwinkel zwischen diesem Vektor und einem Vektor v berechnen
		/// @param v Vektor v
		/// @return Schnittwinkel
		inline float IntersectAngle(CVec3D& v) {	float f = ( ScalarProduct(v) / Length()) * (1.0f / v.Length() ); return acos(f); }	//Schnittwinkel dieses Vektors mit einem anderen Vektor berechnen

		/// Statische Funktion zum Test ob eine Linie einen Kreis schneidet im 3D-Raum
		/// @author David Scherfgen
		inline static bool LineHitsCircle(
					CVec3D& vLineStart,
                    CVec3D& vLineEnd,
                    CVec3D& vCircleCenter,
                    const float fRadius,
                    CVec3D* pvIntersection = 0
		)
		{
			// r vorberechnen
			const float fRadSq = fRadius * fRadius;

			// (p - m) vorberechnen
			CVec3D vPMinusM(vLineStart - vCircleCenter);

			// Wenn der Startpunkt der Linie schon im Kreis liegt,
			// dann brechen wir sofort ab.
			if(vPMinusM.SqrLength() <= fRadSq)
			{
				// Als Schnittpunkt nehmen wir einfach
				// den Startpunkt der Linie.
				if (pvIntersection) *pvIntersection = vLineStart;
				return true;
			}

			// Richtungsvektor der Linie berechnen (u)
			CVec3D vLineDir(vLineEnd - vLineStart);

			// u * (p - m) vorberechnen
			const float fUDotPMinusM = CVec3D::ScalarProduct(vLineDir, vPMinusM);

			// u vorberechnen
			float fLineDirSq = vLineDir.SqrLength();

			// Die Diskriminante berechnen:
			//   (u * (p - m))
			// - (u * ((p - m) - r))
			const float fD = fUDotPMinusM * fUDotPMinusM
							- fLineDirSq * (vPMinusM.SqrLength() - fRadSq);

			// Ist die Diskriminante negativ, null oder positiv?
			if(fD < 0.0f) return false;
			else if(fD < 0.0001f)
			{
				// Die Diskriminante ist (ungefhr) null.
				// Die gesamte Wurzel entfllt und die Lsung ist:
				// (-u * (p - m)) / u.
				// Wir mssen nur noch prfen, ob der Wert
				// im korrekten Bereich [0; 1] liegt.
				const float fS = -fUDotPMinusM / fLineDirSq;
				if(fS < 0.0f || fS > 1.0f) return false;
				else
				{
					// Berhrpunkt!
					if (pvIntersection) *pvIntersection = vLineStart + vLineDir * fS;
					return true;
				}
			}
			else
			{
				// Die Gerade schneidet den Kreis zweimal.
				// Uns interessiert nur die kleinere Lsung fr s,
				// denn das ist die Stelle, wo die Linie den Kreis
				// "zum ersten Mal" schneidet.
				// Diese Lsung berechnen wir nun.
				const float fS = (-fUDotPMinusM - sqrtf(fD)) / fLineDirSq;
				if(fS < 0.0f || fS > 1.0f) return false;
				else
				{
				   if (pvIntersection) *pvIntersection = vLineStart + vLineDir * fS;
					return true;
				}
			}
		}
};

/// Normalisieren eines Vektors
/// D.h. der Betrag d. Vektors = 1
inline void CVec3D::Normalize(void)
{
	float fLen = Length();
	if (!fLen) return;
	fLen = 1.0f / fLen;

	x *= fLen;
	y *= fLen;
	z *= fLen;
}

#endif
